"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.connectToQueryState = void 0;
var _rxjs = require("rxjs");
var _lodash = _interopRequireDefault(require("lodash"));
var _esQuery = require("@kbn/es-query");
var _timefilter = require("../timefilter");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Helper to setup two-way syncing of global data and a state container
 * @param QueryService: either setup or start
 * @param stateContainer to use for syncing
 */
const connectToQueryState = ({
  timefilter: {
    timefilter
  },
  filterManager,
  queryString,
  state$
}, stateContainer, syncConfig) => {
  const syncKeys = [];
  if (syncConfig.time) {
    syncKeys.push('time');
  }
  if (syncConfig.query) {
    syncKeys.push('query');
  }
  if (syncConfig.refreshInterval) {
    syncKeys.push('refreshInterval');
  }
  if (syncConfig.filters) {
    switch (syncConfig.filters) {
      case true:
        syncKeys.push('filters');
        break;
      case _esQuery.FilterStateStore.APP_STATE:
        syncKeys.push('appFilters');
        break;
      case _esQuery.FilterStateStore.GLOBAL_STATE:
        syncKeys.push('globalFilters');
        break;
    }
  }

  // initial syncing
  // TODO:
  // data services take precedence, this seems like a good default,
  // and apps could anyway set their own value after initialisation,
  // but maybe maybe this should be a configurable option?
  const initialState = {
    ...stateContainer.get()
  };
  let initialDirty = false;
  if (syncConfig.time && !_lodash.default.isEqual(initialState.time, timefilter.getTime())) {
    initialState.time = timefilter.getTime();
    initialDirty = true;
  }
  if (syncConfig.refreshInterval && !_lodash.default.isEqual(initialState.refreshInterval, timefilter.getRefreshInterval())) {
    initialState.refreshInterval = timefilter.getRefreshInterval();
    initialDirty = true;
  }
  if (syncConfig.filters) {
    if (syncConfig.filters === true) {
      if (!initialState.filters || !(0, _esQuery.compareFilters)(initialState.filters, filterManager.getFilters(), _esQuery.COMPARE_ALL_OPTIONS)) {
        initialState.filters = filterManager.getFilters();
        initialDirty = true;
      }
    } else if (syncConfig.filters === _esQuery.FilterStateStore.GLOBAL_STATE) {
      if (!initialState.filters || !(0, _esQuery.compareFilters)(initialState.filters, filterManager.getGlobalFilters(), {
        ..._esQuery.COMPARE_ALL_OPTIONS,
        state: false
      })) {
        initialState.filters = filterManager.getGlobalFilters();
        initialDirty = true;
      }
    } else if (syncConfig.filters === _esQuery.FilterStateStore.APP_STATE) {
      if (!initialState.filters || !(0, _esQuery.compareFilters)(initialState.filters, filterManager.getAppFilters(), {
        ..._esQuery.COMPARE_ALL_OPTIONS,
        state: false
      })) {
        initialState.filters = filterManager.getAppFilters();
        initialDirty = true;
      }
    }
  }
  if (initialDirty) {
    stateContainer.set({
      ...stateContainer.get(),
      ...initialState
    });
  }

  // to ignore own state updates
  let updateInProgress = false;
  const subs = [state$.pipe((0, _rxjs.filter)(({
    changes,
    state
  }) => {
    if (updateInProgress) return false;
    return syncKeys.some(syncKey => changes[syncKey]);
  }), (0, _rxjs.map)(({
    changes
  }) => {
    const newState = {};
    if (syncConfig.time && changes.time) {
      newState.time = timefilter.getTime();
    }
    if (syncConfig.query && changes.query) {
      newState.query = queryString.getQuery();
    }
    if (syncConfig.refreshInterval && changes.refreshInterval) {
      newState.refreshInterval = timefilter.getRefreshInterval();
    }
    if (syncConfig.filters) {
      if (syncConfig.filters === true && changes.filters) {
        newState.filters = filterManager.getFilters();
      } else if (syncConfig.filters === _esQuery.FilterStateStore.GLOBAL_STATE && changes.globalFilters) {
        newState.filters = filterManager.getGlobalFilters();
      } else if (syncConfig.filters === _esQuery.FilterStateStore.APP_STATE && changes.appFilters) {
        newState.filters = filterManager.getAppFilters();
      }
    }
    return newState;
  })).subscribe(newState => {
    stateContainer.set({
      ...stateContainer.get(),
      ...newState
    });
  }), stateContainer.state$.subscribe(state => {
    updateInProgress = true;

    // cloneDeep is required because services are mutating passed objects
    // and state in state container is frozen
    if (syncConfig.time) {
      const time = (0, _timefilter.validateTimeRange)(state.time) ? state.time : timefilter.getTimeDefaults();
      if (!_lodash.default.isEqual(time, timefilter.getTime())) {
        timefilter.setTime(_lodash.default.cloneDeep(time));
      }
    }
    if (syncConfig.refreshInterval) {
      const refreshInterval = state.refreshInterval || timefilter.getRefreshIntervalDefaults();
      if (!_lodash.default.isEqual(refreshInterval, timefilter.getRefreshInterval())) {
        timefilter.setRefreshInterval(_lodash.default.cloneDeep(refreshInterval));
      }
    }
    if (syncConfig.query) {
      const curQuery = state.query || queryString.getQuery();
      if (!_lodash.default.isEqual(curQuery, queryString.getQuery())) {
        queryString.setQuery(_lodash.default.cloneDeep(curQuery));
      }
    }
    if (syncConfig.filters) {
      const filters = state.filters || [];
      if (syncConfig.filters === true) {
        if (!(0, _esQuery.compareFilters)(filters, filterManager.getFilters(), _esQuery.COMPARE_ALL_OPTIONS)) {
          filterManager.setFilters(_lodash.default.cloneDeep(filters));
        }
      } else if (syncConfig.filters === _esQuery.FilterStateStore.APP_STATE) {
        if (!(0, _esQuery.compareFilters)(filters, filterManager.getAppFilters(), {
          ..._esQuery.COMPARE_ALL_OPTIONS,
          state: false
        })) {
          filterManager.setAppFilters(_lodash.default.cloneDeep(filters));
        }
      } else if (syncConfig.filters === _esQuery.FilterStateStore.GLOBAL_STATE) {
        if (!(0, _esQuery.compareFilters)(filters, filterManager.getGlobalFilters(), {
          ..._esQuery.COMPARE_ALL_OPTIONS,
          state: false
        })) {
          filterManager.setGlobalFilters(_lodash.default.cloneDeep(filters));
        }
      }
    }
    updateInProgress = false;
  })];
  return () => {
    subs.forEach(s => s.unsubscribe());
  };
};
exports.connectToQueryState = connectToQueryState;