"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UiSettingsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _rxjs = require("rxjs");
var _std = require("@kbn/std");
var _ui_settings_config = require("./ui_settings_config");
var _clients = require("./clients");
var _saved_objects = require("./saved_objects");
var _routes = require("./routes");
var _settings = require("./settings");
var _ui_settings_defaults_client = require("./clients/ui_settings_defaults_client");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/** @internal */
class UiSettingsService {
  constructor(coreContext) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "config$", void 0);
    (0, _defineProperty2.default)(this, "isDist", void 0);
    (0, _defineProperty2.default)(this, "isDev", void 0);
    (0, _defineProperty2.default)(this, "uiSettingsDefaults", new Map());
    (0, _defineProperty2.default)(this, "uiSettingsGlobalDefaults", new Map());
    (0, _defineProperty2.default)(this, "overrides", {});
    (0, _defineProperty2.default)(this, "allowlist", null);
    (0, _defineProperty2.default)(this, "register", (settings = {}) => {
      Object.entries(settings).forEach(([key, value]) => {
        if (this.uiSettingsDefaults.has(key)) {
          throw new Error(`uiSettings for the key [${key}] has been already registered`);
        }
        this.uiSettingsDefaults.set(key, value);
      });
    });
    (0, _defineProperty2.default)(this, "registerGlobal", (settings = {}) => {
      Object.entries(settings).forEach(([key, value]) => {
        if (this.uiSettingsGlobalDefaults.has(key)) {
          throw new Error(`Global uiSettings for the key [${key}] has been already registered`);
        }
        this.uiSettingsGlobalDefaults.set(key, value);
      });
    });
    (0, _defineProperty2.default)(this, "setAllowlist", keys => {
      // Disabling this check for now since it causes some test failures
      // if (this.allowlist) {
      //   throw new Error(
      //     `The uiSettings allowlist has already been set up. Instead of calling setAllowlist(), add your settings to packages/serverless/settings`
      //   );
      // }
      this.allowlist = new Set(keys);
    });
    this.coreContext = coreContext;
    this.log = coreContext.logger.get('ui-settings-service');
    this.isDist = coreContext.env.packageInfo.dist;
    this.config$ = coreContext.configService.atPath(_ui_settings_config.uiSettingsConfig.path);
    this.isDev = coreContext.env.mode.dev;
  }
  async preboot() {
    this.log.debug('Prebooting ui settings service');
    const {
      overrides
    } = await (0, _rxjs.firstValueFrom)(this.config$);
    this.overrides = overrides;
    this.register((0, _settings.getCoreSettings)({
      isDist: this.isDist
    }));
    return {
      createDefaultsClient: () => new _ui_settings_defaults_client.UiSettingsDefaultsClient({
        defaults: (0, _std.mapToObject)(this.uiSettingsDefaults),
        overrides: this.overrides,
        log: this.log.get('core defaults')
      })
    };
  }
  async setup({
    http,
    savedObjects
  }) {
    this.log.debug('Setting up ui settings service');
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    this.overrides = config.overrides;
    savedObjects.registerType(_saved_objects.uiSettingsType);
    savedObjects.registerType(_saved_objects.uiSettingsGlobalType);
    const router = http.createRouter('');
    (0, _routes.registerInternalRoutes)(router);

    // Register public routes by default unless the publicApiEnabled config setting is set to false
    if (!config.hasOwnProperty('publicApiEnabled') || config.publicApiEnabled === true) {
      (0, _routes.registerRoutes)(router);
    }
    return {
      register: this.register,
      registerGlobal: this.registerGlobal,
      setAllowlist: this.setAllowlist
    };
  }
  async start() {
    if (this.allowlist) {
      // If we are in development mode, check if all settings in the allowlist are registered
      if (this.isDev) {
        this.validateAllowlist();
      }
      this.applyAllowlist(this.uiSettingsDefaults, false);
      this.applyAllowlist(this.uiSettingsGlobalDefaults, true);
    }
    this.validatesDefinitions();
    this.validatesOverrides();
    return {
      asScopedToClient: this.getScopedClientFactory('namespace'),
      globalAsScopedToClient: this.getScopedClientFactory('global')
    };
  }
  async stop() {}
  getScopedClientFactory(scope) {
    const {
      version,
      buildNum
    } = this.coreContext.env.packageInfo;
    return savedObjectsClient => {
      const isNamespaceScope = scope === 'namespace';
      const options = {
        type: isNamespaceScope ? 'config' : 'config-global',
        id: (0, _std.stripVersionQualifier)(version),
        buildNum,
        savedObjectsClient,
        defaults: isNamespaceScope ? (0, _std.mapToObject)(this.uiSettingsDefaults) : (0, _std.mapToObject)(this.uiSettingsGlobalDefaults),
        overrides: isNamespaceScope ? this.overrides : {},
        log: this.log
      };
      return _clients.UiSettingsClientFactory.create(options);
    };
  }
  validateAllowlist() {
    var _this$allowlist;
    (_this$allowlist = this.allowlist) === null || _this$allowlist === void 0 ? void 0 : _this$allowlist.forEach(key => {
      if (!this.uiSettingsDefaults.has(key) && !this.uiSettingsGlobalDefaults.has(key)) {
        throw new Error(`The uiSetting with key [${key}] is in the allowlist but is not registered. Make sure to remove it from the allowlist in /packages/serverless/settings`);
      }
    });
  }
  setReadonlyMode(key, mode, isGlobal) {
    if (isGlobal) {
      const definition = this.uiSettingsGlobalDefaults.get(key);
      if (definition) {
        this.uiSettingsGlobalDefaults.set(key, {
          ...definition,
          readonlyMode: mode
        });
      }
    } else {
      const definition = this.uiSettingsDefaults.get(key);
      if (definition) {
        this.uiSettingsDefaults.set(key, {
          ...definition,
          readonlyMode: mode
        });
      }
    }
  }
  applyAllowlist(settingsDefaults, isGlobal) {
    for (const [key, definition] of settingsDefaults) {
      var _this$allowlist2, _this$allowlist3;
      // Settings in the allowlist that are already read-only should have 'ui' readonly mode
      if ((_this$allowlist2 = this.allowlist) !== null && _this$allowlist2 !== void 0 && _this$allowlist2.has(key) && definition.readonly === true) {
        this.setReadonlyMode(key, 'ui', isGlobal);
      }

      // Setting that are not in the allowlist should have 'strict' readonly mode
      if (!((_this$allowlist3 = this.allowlist) !== null && _this$allowlist3 !== void 0 && _this$allowlist3.has(key))) {
        definition.readonly = true;
        this.setReadonlyMode(key, 'strict', isGlobal);
      }
    }
  }
  validatesDefinitions() {
    for (const [key, definition] of this.uiSettingsDefaults) {
      if (!definition.schema) {
        throw new Error(`Validation schema is not provided for [${key}] UI Setting`);
      }
      definition.schema.validate(definition.value, {}, `ui settings defaults [${key}]`);
    }
    for (const [key, definition] of this.uiSettingsGlobalDefaults) {
      if (!definition.schema) {
        throw new Error(`Validation schema is not provided for [${key}] Global UI Setting`);
      }
      definition.schema.validate(definition.value, {});
    }
  }
  validatesOverrides() {
    for (const [key, value] of Object.entries(this.overrides)) {
      const definition = this.uiSettingsDefaults.get(key);
      // overrides might contain UiSettings for a disabled plugin
      if (definition !== null && definition !== void 0 && definition.schema) {
        definition.schema.validate(value, {}, `ui settings overrides [${key}]`);
      }
    }
  }
}
exports.UiSettingsService = UiSettingsService;