"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.logPluginsStatusChanges = void 0;
var _lodash = require("lodash");
var _rxjs = require("rxjs");
var _coreStatusCommon = require("@kbn/core-status-common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

// let plugins log up to 3 status changes every 30s (extra messages will be throttled / aggregated)
const MAX_MESSAGES_PER_PLUGIN_PER_INTERVAL = 3;
const THROTTLE_INTERVAL_MILLIS = 30000;
const MAX_THROTTLED_MESSAGES = 10;
const logPluginsStatusChanges = ({
  logger,
  plugins$,
  stop$,
  maxMessagesPerPluginPerInterval = MAX_MESSAGES_PER_PLUGIN_PER_INTERVAL,
  throttleIntervalMillis = THROTTLE_INTERVAL_MILLIS,
  maxThrottledMessages = MAX_THROTTLED_MESSAGES
}) => {
  const buffer = new _rxjs.Subject();
  const throttled$ = buffer.asObservable().pipe((0, _rxjs.takeUntil)(stop$), (0, _rxjs.bufferTime)(maxMessagesPerPluginPerInterval), (0, _rxjs.map)(statuses => {
    const aggregated =
    // aggregate repeated messages, and count nbr. of repetitions
    statuses.filter((candidateStatus, index) => {
      const firstMessageIndex = statuses.findIndex(status => candidateStatus.name === status.name && candidateStatus.level === status.level && candidateStatus.summary === status.summary);
      if (index !== firstMessageIndex) {
        var _statuses$firstMessag;
        // this is not the first time this message is logged, increase 'repeats' counter for the first occurrence
        statuses[firstMessageIndex].repeats = ((_statuses$firstMessag = statuses[firstMessageIndex].repeats) !== null && _statuses$firstMessag !== void 0 ? _statuses$firstMessag : 1) + 1;
        return false;
      } else {
        // this is the first time this message is logged, let it through
        return true;
      }
    });
    if (aggregated.length > maxThrottledMessages) {
      const list = (0, _lodash.uniq)(aggregated.slice(maxThrottledMessages).map(({
        name
      }) => name)).join(', ');
      return [...aggregated.slice(0, maxThrottledMessages), `${aggregated.length - maxThrottledMessages} other status updates from [${list}] have been truncated to avoid flooding the logs`];
    } else {
      return aggregated;
    }
  }), (0, _rxjs.concatAll)());
  const lastMessagesTimestamps = {};
  const direct$ = plugins$.pipe((0, _rxjs.startWith)({}),
  // consider all plugins unavailable by default
  (0, _rxjs.takeUntil)(stop$), (0, _rxjs.pairwise)(), (0, _rxjs.map)(([oldStatus, newStatus]) => getPluginUpdates(oldStatus, newStatus)), (0, _rxjs.concatAll)(), (0, _rxjs.filter)(pluginStatus => {
    const now = Date.now();
    const pluginQuota = lastMessagesTimestamps[pluginStatus.name] || [];
    lastMessagesTimestamps[pluginStatus.name] = pluginQuota;

    // remove timestamps of messages older than the threshold
    while (pluginQuota.length > 0 && pluginQuota[0] < now - throttleIntervalMillis) {
      pluginQuota.shift();
    }
    if (pluginQuota.length >= maxMessagesPerPluginPerInterval) {
      // we're still over quota, throttle the message
      buffer.next(pluginStatus);
      return false;
    } else {
      // let the message pass through
      pluginQuota.push(now);
      return true;
    }
  }));
  return (0, _rxjs.merge)(direct$, throttled$).subscribe(event => {
    if (typeof event === 'string') {
      logger.warn(event);
    } else {
      const pluginStatus = event;
      const {
        name
      } = pluginStatus;
      const pluginLogger = logger.get(name);
      const message = getPluginStatusMessage(pluginStatus);
      switch (pluginStatus.level) {
        case _coreStatusCommon.ServiceStatusLevels.available:
          pluginLogger.info(message);
          break;
        case _coreStatusCommon.ServiceStatusLevels.degraded:
          pluginLogger.warn(message);
          break;
        default:
          pluginLogger.error(message);
      }
    }
  });
};
exports.logPluginsStatusChanges = logPluginsStatusChanges;
const getPluginUpdates = (previous, next) => Object.entries(next).filter(([name, pluginStatus]) => {
  var _previous$name;
  const currentLevel = pluginStatus.level;
  const previousLevel = (_previous$name = previous[name]) === null || _previous$name === void 0 ? void 0 : _previous$name.level;
  return pluginStatus.reported && currentLevel !== previousLevel;
}).map(([name, pluginStatus]) => ({
  ...pluginStatus,
  name
}));
const getPluginStatusMessage = ({
  name,
  level,
  summary,
  detail,
  repeats = 0
}) => `${name} plugin is now ${level === null || level === void 0 ? void 0 : level.toString()}: ${summary}${detail ? ` | ${detail}` : ''}${repeats > 1 ? ` (repeated ${repeats} times)` : ''}`;