"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.init = void 0;
var _lodash = require("lodash");
var Either = _interopRequireWildcard(require("fp-ts/lib/Either"));
var _retry_state = require("../../../model/retry_state");
var _helpers = require("../../../model/helpers");
var _actions = require("../../actions");
var _utils = require("../../utils");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const init = (state, res, context) => {
  if (Either.isLeft(res)) {
    const left = res.left;
    if ((0, _actions.isTypeof)(left, 'incompatible_cluster_routing_allocation')) {
      const retryErrorMessage = `[${left.type}] Incompatible Elasticsearch cluster settings detected. Remove the persistent and transient Elasticsearch cluster setting 'cluster.routing.allocation.enable' or set it to a value of 'all' to allow migrations to proceed. Refer to ${context.migrationDocLinks.routingAllocationDisabled} for more information on how to resolve the issue.`;
      return (0, _retry_state.delayRetryState)(state, retryErrorMessage, context.maxRetryAttempts);
    } else {
      return (0, _helpers.throwBadResponse)(state, left);
    }
  }
  const types = context.types.map(type => context.typeRegistry.getType(type));
  const logs = [...state.logs];
  const indices = res.right;
  const aliasesRes = (0, _helpers.getAliases)(indices);
  if (Either.isLeft(aliasesRes)) {
    return {
      ...state,
      controlState: 'FATAL',
      reason: `The ${aliasesRes.left.alias} alias is pointing to multiple indices: ${aliasesRes.left.indices.join(',')}.`
    };
  }
  const aliasMap = aliasesRes.right;
  const currentIndex = (0, _utils.getCurrentIndex)({
    indices: Object.keys(indices),
    aliases: aliasMap,
    indexPrefix: context.indexPrefix
  });

  // No indices were found, likely because it is a fresh cluster.
  // In that case, we just create the index.
  if (!currentIndex) {
    return {
      ...state,
      logs,
      controlState: 'CREATE_TARGET_INDEX',
      currentIndex: `${context.indexPrefix}_1`,
      indexMappings: (0, _utils.buildIndexMappings)({
        types
      }),
      creationAliases: (0, _utils.getCreationAliases)({
        indexPrefix: context.indexPrefix,
        kibanaVersion: context.kibanaVersion
      })
    };
  }

  // Index was found. This is the standard scenario, we check the model versions
  // compatibility before going further.
  const currentMappings = indices[currentIndex].mappings;

  // Index is already present, so we check which algo was last used on it
  const currentAlgo = (0, _utils.checkIndexCurrentAlgorithm)(currentMappings);
  logs.push({
    level: 'info',
    message: `INIT: current algo check result: ${currentAlgo}`
  });

  // incompatible (pre 8.8/index-split https://github.com/elastic/kibana/pull/154888) v2 algo => we terminate
  if (currentAlgo === 'v2-incompatible') {
    return {
      ...state,
      logs,
      controlState: 'FATAL',
      reason: `Index ${currentIndex} is using an incompatible version of the v2 algorithm`
    };
  }
  // unknown algo => we terminate
  if (currentAlgo === 'unknown') {
    return {
      ...state,
      logs,
      controlState: 'FATAL',
      reason: `Cannot identify algorithm used for index ${currentIndex}`
    };
  }
  const existingAliases = Object.keys(indices[currentIndex].aliases);
  const aliasActions = (0, _utils.getAliasActions)({
    existingAliases,
    currentIndex,
    indexPrefix: context.indexPrefix,
    kibanaVersion: context.kibanaVersion
  });
  // cloning as we may be mutating it in later stages.
  let currentIndexMeta = (0, _lodash.cloneDeep)(currentMappings._meta);
  if (currentAlgo === 'v2-compatible' || currentAlgo === 'v2-partially-migrated') {
    currentIndexMeta = (0, _utils.removePropertiesFromV2)(currentIndexMeta);
  }
  const commonState = {
    logs,
    currentIndex,
    currentIndexMeta,
    aliases: existingAliases,
    aliasActions,
    previousMappings: currentMappings,
    previousAlgorithm: currentAlgo === 'v2-compatible' || currentAlgo === 'v2-partially-migrated' ? 'v2' : 'zdt'
  };

  // compatible (8.8+) v2 algo => we jump to update index mapping
  if (currentAlgo === 'v2-compatible') {
    const indexMappings = (0, _utils.buildIndexMappings)({
      types
    });
    return {
      ...state,
      controlState: 'UPDATE_INDEX_MAPPINGS',
      ...commonState,
      additiveMappingChanges: indexMappings.properties
    };
  }

  // Index was found and is already using ZDT algo. This is the standard scenario.
  // We check the model versions compatibility before going further.
  const versionCheck = (0, _utils.checkVersionCompatibility)({
    mappings: currentMappings,
    types,
    source: 'mappingVersions',
    deletedTypes: context.deletedTypes
  });
  logs.push({
    level: 'info',
    message: `INIT: mapping version check result: ${versionCheck.status}`
  });
  switch (versionCheck.status) {
    // app version is greater than the index mapping version.
    // scenario of an upgrade: we need to update the mappings
    case 'greater':
      const additiveMappingChanges = (0, _utils.generateAdditiveMappingDiff)({
        types,
        mapping: currentMappings,
        deletedTypes: context.deletedTypes
      });
      return {
        ...state,
        controlState: 'UPDATE_INDEX_MAPPINGS',
        ...commonState,
        additiveMappingChanges
      };
    // app version and index mapping version are the same.
    // either application upgrade without model change, or a simple reboot on the same version.
    // In that case we jump directly to alias update
    case 'equal':
      return {
        ...state,
        controlState: aliasActions.length ? 'UPDATE_ALIASES' : 'INDEX_STATE_UPDATE_DONE',
        ...commonState
      };
    // app version is lower than the index mapping version.
    // either a rollback scenario, or an old node rebooting during the cohabitation period
    // in that case, we simply no-op the expand phase.
    case 'lesser':
      return {
        ...state,
        controlState: 'INDEX_STATE_UPDATE_DONE',
        ...commonState
      };
    // conflicts: version for some types are greater, some are lower
    // shouldn't occur in any normal scenario - cannot recover
    case 'conflict':
    default:
      return {
        ...state,
        controlState: 'FATAL',
        reason: 'Model version conflict: inconsistent higher/lower versions',
        logs
      };
  }
};
exports.init = init;