"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.performBulkUpdate = void 0;
var _coreElasticsearchServerInternal = require("@kbn/core-elasticsearch-server-internal");
var _coreSavedObjectsServer = require("@kbn/core-saved-objects-server");
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _coreSavedObjectsBaseServerInternal = require("@kbn/core-saved-objects-base-server-internal");
var _constants = require("../constants");
var _utils = require("./utils");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const performBulkUpdate = async ({
  objects,
  options
}, {
  registry,
  helpers,
  allowedTypes,
  client,
  serializer,
  extensions = {}
}) => {
  const {
    common: commonHelper,
    encryption: encryptionHelper,
    migration: migrationHelper
  } = helpers;
  const {
    securityExtension
  } = extensions;
  const {
    migrationVersionCompatibility
  } = options;
  const namespace = commonHelper.getCurrentNamespace(options.namespace);
  const time = (0, _utils.getCurrentTime)();
  let bulkGetRequestIndexCounter = 0;
  const expectedBulkGetResults = objects.map(object => {
    const {
      type,
      id,
      attributes,
      references,
      version,
      namespace: objectNamespace,
      mergeAttributes = true
    } = object;
    let error;
    if (!allowedTypes.includes(type)) {
      error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id);
    } else {
      try {
        if (objectNamespace === _coreSavedObjectsUtilsServer.ALL_NAMESPACES_STRING) {
          error = _coreSavedObjectsServer.SavedObjectsErrorHelpers.createBadRequestError('"namespace" cannot be "*"');
        }
      } catch (e) {
        error = e;
      }
    }
    if (error) {
      return (0, _utils.left)({
        id,
        type,
        error: (0, _utils.errorContent)(error)
      });
    }
    const documentToSave = {
      [type]: attributes,
      updated_at: time,
      ...(Array.isArray(references) && {
        references
      })
    };
    return (0, _utils.right)({
      type,
      id,
      version,
      documentToSave,
      objectNamespace,
      esRequestIndex: bulkGetRequestIndexCounter++,
      migrationVersionCompatibility,
      mergeAttributes
    });
  });
  const validObjects = expectedBulkGetResults.filter(_utils.isRight);
  if (validObjects.length === 0) {
    return {
      // Technically the returned array should only contain SavedObject results, but for errors this is not true (we cast to 'any' below)
      saved_objects: expectedBulkGetResults.map(({
        value
      }) => value)
    };
  }

  // `objectNamespace` is a namespace string, while `namespace` is a namespace ID.
  // The object namespace string, if defined, will supersede the operation's namespace ID.
  const namespaceString = _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(namespace);
  const getNamespaceId = objectNamespace => objectNamespace !== undefined ? _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceStringToId(objectNamespace) : namespace;
  const getNamespaceString = objectNamespace => objectNamespace !== null && objectNamespace !== void 0 ? objectNamespace : namespaceString;
  const bulkGetDocs = validObjects.map(({
    value: {
      type,
      id,
      objectNamespace
    }
  }) => ({
    _id: serializer.generateRawId(getNamespaceId(objectNamespace), type, id),
    _index: commonHelper.getIndexForType(type),
    _source: true
  }));
  const bulkGetResponse = bulkGetDocs.length ? await client.mget({
    body: {
      docs: bulkGetDocs
    }
  }, {
    ignore: [404],
    meta: true
  }) : undefined;
  // fail fast if we can't verify a 404 response is from Elasticsearch
  if (bulkGetResponse && (0, _coreElasticsearchServerInternal.isNotFoundFromUnsupportedServer)({
    statusCode: bulkGetResponse.statusCode,
    headers: bulkGetResponse.headers
  })) {
    throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundEsUnavailableError();
  }
  const authObjects = validObjects.map(element => {
    const {
      type,
      id,
      objectNamespace,
      esRequestIndex: index
    } = element.value;
    const preflightResult = bulkGetResponse.body.docs[index];
    if (registry.isMultiNamespace(type)) {
      var _preflightResult$_sou, _preflightResult$_sou2;
      return {
        type,
        id,
        objectNamespace,
        // @ts-expect-error MultiGetHit._source is optional
        existingNamespaces: (_preflightResult$_sou = (_preflightResult$_sou2 = preflightResult._source) === null || _preflightResult$_sou2 === void 0 ? void 0 : _preflightResult$_sou2.namespaces) !== null && _preflightResult$_sou !== void 0 ? _preflightResult$_sou : []
      };
    } else {
      return {
        type,
        id,
        objectNamespace,
        existingNamespaces: []
      };
    }
  });
  const authorizationResult = await (securityExtension === null || securityExtension === void 0 ? void 0 : securityExtension.authorizeBulkUpdate({
    namespace,
    objects: authObjects
  }));
  let bulkUpdateRequestIndexCounter = 0;
  const bulkUpdateParams = [];
  const expectedBulkUpdateResults = await Promise.all(expectedBulkGetResults.map(async expectedBulkGetResult => {
    if ((0, _utils.isLeft)(expectedBulkGetResult)) {
      return expectedBulkGetResult;
    }
    const {
      esRequestIndex,
      id,
      type,
      version,
      documentToSave,
      objectNamespace,
      mergeAttributes
    } = expectedBulkGetResult.value;
    let namespaces;
    const versionProperties = (0, _utils.getExpectedVersionProperties)(version);
    const indexFound = (bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.statusCode) !== 404;
    const actualResult = indexFound ? bulkGetResponse === null || bulkGetResponse === void 0 ? void 0 : bulkGetResponse.body.docs[esRequestIndex] : undefined;
    const docFound = indexFound && (0, _utils.isMgetDoc)(actualResult) && actualResult.found;
    const isMultiNS = registry.isMultiNamespace(type);
    if (!docFound || isMultiNS && !(0, _utils.rawDocExistsInNamespace)(registry, actualResult, getNamespaceId(objectNamespace))) {
      return (0, _utils.left)({
        id,
        type,
        error: (0, _utils.errorContent)(_coreSavedObjectsServer.SavedObjectsErrorHelpers.createGenericNotFoundError(type, id))
      });
    }
    if (isMultiNS) {
      var _source$namespaces;
      // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
      namespaces = (_source$namespaces = actualResult._source.namespaces) !== null && _source$namespaces !== void 0 ? _source$namespaces : [
      // @ts-expect-error MultiGetHit is incorrectly missing _id, _source
      _coreSavedObjectsUtilsServer.SavedObjectsUtils.namespaceIdToString(actualResult._source.namespace)];
    } else if (registry.isSingleNamespace(type)) {
      // if `objectNamespace` is undefined, fall back to `options.namespace`
      namespaces = [getNamespaceString(objectNamespace)];
    }
    const document = (0, _utils.getSavedObjectFromSource)(registry, type, id, actualResult, {
      migrationVersionCompatibility
    });
    let migrated;
    try {
      migrated = migrationHelper.migrateStorageDocument(document);
    } catch (migrateStorageDocError) {
      throw _coreSavedObjectsServer.SavedObjectsErrorHelpers.decorateGeneralError(migrateStorageDocError, 'Failed to migrate document to the latest version.');
    }
    const typeDefinition = registry.getType(type);
    const encryptedUpdatedAttributes = await encryptionHelper.optionallyEncryptAttributes(type, id, objectNamespace || namespace, documentToSave[type]);
    const updatedAttributes = mergeAttributes ? (0, _utils.mergeForUpdate)({
      targetAttributes: {
        ...migrated.attributes
      },
      updatedAttributes: encryptedUpdatedAttributes,
      typeMappings: typeDefinition.mappings
    }) : encryptedUpdatedAttributes;
    const migratedUpdatedSavedObjectDoc = migrationHelper.migrateInputDocument({
      ...migrated,
      id,
      type,
      namespace,
      namespaces,
      attributes: updatedAttributes,
      updated_at: time,
      ...(Array.isArray(documentToSave.references) && {
        references: documentToSave.references
      })
    });
    const updatedMigratedDocumentToSave = serializer.savedObjectToRaw(migratedUpdatedSavedObjectDoc);
    const expectedResult = {
      type,
      id,
      namespaces,
      esRequestIndex: bulkUpdateRequestIndexCounter++,
      documentToSave: expectedBulkGetResult.value.documentToSave,
      rawMigratedUpdatedDoc: updatedMigratedDocumentToSave,
      migrationVersionCompatibility
    };
    bulkUpdateParams.push({
      index: {
        _id: serializer.generateRawId(getNamespaceId(objectNamespace), type, id),
        _index: commonHelper.getIndexForType(type),
        ...versionProperties
      }
    }, updatedMigratedDocumentToSave._source);
    return (0, _utils.right)(expectedResult);
  }));
  const {
    refresh = _constants.DEFAULT_REFRESH_SETTING
  } = options;
  const bulkUpdateResponse = bulkUpdateParams.length ? await client.bulk({
    refresh,
    body: bulkUpdateParams,
    _source_includes: ['originId'],
    require_alias: true
  }) : undefined;
  const result = {
    saved_objects: expectedBulkUpdateResults.map(expectedResult => {
      var _bulkUpdateResponse$i;
      if ((0, _utils.isLeft)(expectedResult)) {
        return expectedResult.value;
      }
      const {
        type,
        id,
        namespaces,
        documentToSave,
        esRequestIndex,
        rawMigratedUpdatedDoc
      } = expectedResult.value;
      const response = (_bulkUpdateResponse$i = bulkUpdateResponse === null || bulkUpdateResponse === void 0 ? void 0 : bulkUpdateResponse.items[esRequestIndex]) !== null && _bulkUpdateResponse$i !== void 0 ? _bulkUpdateResponse$i : {};
      const rawResponse = Object.values(response)[0];
      const error = (0, _utils.getBulkOperationError)(type, id, rawResponse);
      if (error) {
        return {
          type,
          id,
          error
        };
      }
      const {
        _seq_no: seqNo,
        _primary_term: primaryTerm
      } = rawResponse;

      // eslint-disable-next-line @typescript-eslint/naming-convention
      const {
        [type]: attributes,
        references,
        updated_at
      } = documentToSave;
      const {
        originId
      } = rawMigratedUpdatedDoc._source;
      return {
        id,
        type,
        ...(namespaces && {
          namespaces
        }),
        ...(originId && {
          originId
        }),
        updated_at,
        version: (0, _coreSavedObjectsBaseServerInternal.encodeVersion)(seqNo, primaryTerm),
        attributes,
        references
      };
    })
  };
  return encryptionHelper.optionallyDecryptAndRedactBulkResult(result, authorizationResult === null || authorizationResult === void 0 ? void 0 : authorizationResult.typeMap, objects);
};
exports.performBulkUpdate = performBulkUpdate;