"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CoreAppsService = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _querystring = require("querystring");
var _configSchema = require("@kbn/config-schema");
var _repoInfo = require("@kbn/repo-info");
var _rxjs = require("rxjs");
var _core_app_config = require("./core_app_config");
var _bundle_routes = require("./bundle_routes");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/** @internal */

/** @internal */
class CoreAppsService {
  constructor(core) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "env", void 0);
    (0, _defineProperty2.default)(this, "configService", void 0);
    (0, _defineProperty2.default)(this, "config$", void 0);
    this.logger = core.logger.get('core-app');
    this.env = core.env;
    this.configService = core.configService;
    this.config$ = this.configService.atPath(_core_app_config.CoreAppPath).pipe((0, _rxjs.map)(rawCfg => new _core_app_config.CoreAppConfig(rawCfg)));
  }
  preboot(corePreboot, uiPlugins) {
    this.logger.debug('Prebooting core app.');

    // We register app-serving routes only if there are `preboot` plugins that may need them.
    if (uiPlugins.public.size > 0) {
      this.registerPrebootDefaultRoutes(corePreboot, uiPlugins);
      this.registerStaticDirs(corePreboot, uiPlugins);
    }
  }
  async setup(coreSetup, uiPlugins) {
    this.logger.debug('Setting up core app.');
    const config = await (0, _rxjs.firstValueFrom)(this.config$);
    this.registerDefaultRoutes(coreSetup, uiPlugins, config);
    this.registerStaticDirs(coreSetup, uiPlugins);
  }
  registerPrebootDefaultRoutes(corePreboot, uiPlugins) {
    corePreboot.http.registerRoutes('', router => {
      this.registerCommonDefaultRoutes({
        basePath: corePreboot.http.basePath,
        httpResources: corePreboot.httpResources.createRegistrar(router),
        staticAssets: corePreboot.http.staticAssets,
        router,
        uiPlugins,
        onResourceNotFound: async (req, res) =>
        // The API consumers might call various Kibana APIs (e.g. `/api/status`) when Kibana is still at the preboot
        // stage, and the main HTTP server that registers API handlers isn't up yet. At this stage we don't know if
        // the API endpoint exists or not, and hence cannot reply with `404`. We also should not reply with completely
        // unexpected response (`200 text/html` for the Core app). The only suitable option is to reply with `503`
        // like we do for all other unknown non-GET requests at the preboot stage.
        req.route.path.startsWith('/api/') || req.route.path.startsWith('/internal/') ? res.customError({
          statusCode: 503,
          headers: {
            'Retry-After': '30'
          },
          body: 'Kibana server is not ready yet',
          bypassErrorFormat: true
        }) : res.renderCoreApp()
      });
    });
  }
  registerDefaultRoutes(coreSetup, uiPlugins, config) {
    const httpSetup = coreSetup.http;
    const router = httpSetup.createRouter('');
    const resources = coreSetup.httpResources.createRegistrar(router);
    router.get({
      path: '/',
      validate: false,
      options: {
        access: 'public'
      }
    }, async (context, req, res) => {
      const {
        uiSettings
      } = await context.core;
      const defaultRoute = await uiSettings.client.get('defaultRoute');
      const basePath = httpSetup.basePath.get(req);
      const url = `${basePath}${defaultRoute}`;
      return res.redirected({
        headers: {
          location: url
        }
      });
    });
    this.registerCommonDefaultRoutes({
      basePath: coreSetup.http.basePath,
      httpResources: resources,
      staticAssets: coreSetup.http.staticAssets,
      router,
      uiPlugins,
      onResourceNotFound: async (req, res) => res.notFound()
    });
    resources.register({
      path: '/app/{id}/{any*}',
      validate: false,
      options: {
        authRequired: true
      }
    }, async (context, request, response) => {
      return response.renderCoreApp();
    });
    const anonymousStatusPage = coreSetup.status.isStatusPageAnonymous();
    resources.register({
      path: '/status',
      validate: false,
      options: {
        authRequired: !anonymousStatusPage
      }
    }, async (context, request, response) => {
      if (anonymousStatusPage) {
        return response.renderAnonymousCoreApp();
      } else {
        return response.renderCoreApp();
      }
    });
    if (config.allowDynamicConfigOverrides) {
      this.registerInternalCoreSettingsRoute(router);
    }
  }

  /**
   * Registers the HTTP API that allows updating in-memory the settings that opted-in to be dynamically updatable.
   * @param router {@link IRouter}
   * @private
   */
  registerInternalCoreSettingsRoute(router) {
    router.versioned.put({
      path: '/internal/core/_settings',
      access: 'internal',
      options: {
        tags: ['access:updateDynamicConfig']
      }
    }).addVersion({
      version: '1',
      validate: {
        request: {
          body: _configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any())
        },
        response: {
          '200': {
            body: () => _configSchema.schema.object({
              ok: _configSchema.schema.boolean()
            })
          }
        }
      }
    }, async (context, req, res) => {
      try {
        this.configService.setDynamicConfigOverrides(req.body);
      } catch (err) {
        if (err instanceof _configSchema.ValidationError) {
          return res.badRequest({
            body: err
          });
        }
        throw err;
      }
      return res.ok({
        body: {
          ok: true
        }
      });
    });
  }
  registerCommonDefaultRoutes({
    router,
    basePath,
    staticAssets,
    uiPlugins,
    onResourceNotFound,
    httpResources
  }) {
    // catch-all route
    httpResources.register({
      path: '/{path*}',
      validate: {
        params: _configSchema.schema.object({
          path: _configSchema.schema.maybe(_configSchema.schema.string())
        }),
        query: _configSchema.schema.maybe(_configSchema.schema.recordOf(_configSchema.schema.string(), _configSchema.schema.any()))
      }
    }, async (context, req, res) => {
      const {
        query,
        params
      } = req;
      const {
        path
      } = params;
      if (!path || !path.endsWith('/') || path.startsWith('/')) {
        return onResourceNotFound(req, res);
      }

      // remove trailing slash
      const requestBasePath = basePath.get(req);
      let rewrittenPath = path.slice(0, -1);
      if (`/${path}`.startsWith(requestBasePath)) {
        rewrittenPath = rewrittenPath.substring(requestBasePath.length);
      }
      const querystring = query ? (0, _querystring.stringify)(query) : undefined;
      const url = `${requestBasePath}/${encodeURIComponent(rewrittenPath)}${querystring ? `?${querystring}` : ''}`;
      return res.redirected({
        headers: {
          location: url
        }
      });
    });
    router.get({
      path: '/core',
      validate: false
    }, async (context, req, res) => res.ok({
      body: {
        version: '0.0.1'
      }
    }));
    (0, _bundle_routes.registerBundleRoutes)({
      router,
      uiPlugins,
      staticAssets,
      packageInfo: this.env.packageInfo
    });
  }

  // After the package is built and bootstrap extracts files to bazel-bin,
  // assets are exposed at the root of the package and in the package's node_modules dir
  registerStaticDirs(core, uiPlugins) {
    /**
     * Serve UI from sha-scoped and not-sha-scoped paths to allow time for plugin code to migrate
     * Eventually we only want to serve from the sha scoped path
     */
    [core.http.staticAssets.prependServerPath('/ui/{path*}'), '/ui/{path*}'].forEach(path => {
      core.http.registerStaticDir(path, (0, _repoInfo.fromRoot)('node_modules/@kbn/core-apps-server-internal/assets'));
    });
    for (const [pluginName, pluginInfo] of uiPlugins.internal) {
      if (!pluginInfo.publicAssetsDir) continue;
      /**
       * Serve UI from sha-scoped and not-sha-scoped paths to allow time for plugin code to migrate
       * Eventually we only want to serve from the sha scoped path
       */
      [core.http.staticAssets.getPluginServerPath(pluginName, '{path*}'), `/plugins/${pluginName}/assets/{path*}`].forEach(path => {
        core.http.registerStaticDir(path, pluginInfo.publicAssetsDir);
      });
    }
  }
}
exports.CoreAppsService = CoreAppsService;