"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getSearchSuggestionsAggregationBuilder = void 0;
var _common = require("@kbn/data-views-plugin/common");
var _lodash = require("lodash");
var _ip_search = require("../../../common/options_list/ip_search");
var _is_valid_search = require("../../../common/options_list/is_valid_search");
var _suggestions_searching = require("../../../common/options_list/suggestions_searching");
var _options_list_exact_match = require("./options_list_exact_match");
var _options_list_suggestion_query_helpers = require("./options_list_suggestion_query_helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Type-specific search suggestion aggregations. These queries are highly impacted by the field type.
 */
const getSearchSuggestionsAggregationBuilder = request => {
  const {
    fieldSpec
  } = request;

  // note that date and boolean fields are non-searchable, so type-specific search aggs are not necessary;
  // number fields, on the other hand, only support exact match searching - so, this also does not need a
  // type-specific agg because it will be handled by `exactMatchSearchAggregation`
  switch (fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type) {
    case 'ip':
      {
        return suggestionAggSubtypes.ip;
      }
    case 'string':
      {
        return suggestionAggSubtypes.textOrKeywordOrNested;
      }
    default:
      // safe guard just in case an invalid/unsupported field type somehow got through
      return (0, _options_list_exact_match.getExactMatchAggregationBuilder)();
  }
};
exports.getSearchSuggestionsAggregationBuilder = getSearchSuggestionsAggregationBuilder;
const suggestionAggSubtypes = {
  /**
   * The "textOrKeywordOrNested" query / parser should be used whenever the field is built on some type of string field,
   * regardless of if it is keyword only, keyword+text, or some nested keyword/keyword+text field.
   */
  textOrKeywordOrNested: {
    buildAggregation: ({
      searchTechnique,
      searchString,
      fieldName,
      fieldSpec,
      sort,
      size
    }) => {
      var _fieldSpec$type;
      const hasSearchString = searchString && searchString.length > 0;
      if (!hasSearchString || (fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type) === 'date') {
        // we can assume that this is only ever called with a search string, and date fields are not
        // currently searchable; so, if any of these things is true, this is invalid.
        return undefined;
      }
      const subTypeNested = fieldSpec && (0, _common.getFieldSubtypeNested)(fieldSpec);
      let textOrKeywordQuery = {
        filteredSuggestions: {
          filter: {
            [searchTechnique !== null && searchTechnique !== void 0 ? searchTechnique : (0, _suggestions_searching.getDefaultSearchTechnique)((_fieldSpec$type = fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type) !== null && _fieldSpec$type !== void 0 ? _fieldSpec$type : 'string')]: {
              [fieldName]: {
                value: searchTechnique === 'wildcard' ? `*${(0, _options_list_suggestion_query_helpers.getEscapedWildcardQuery)(searchString)}*` : searchString,
                case_insensitive: true
              }
            }
          },
          aggs: {
            suggestions: {
              terms: {
                size,
                field: fieldName,
                shard_size: 10,
                order: (0, _options_list_suggestion_query_helpers.getSortType)(sort)
              }
            },
            unique_terms: {
              cardinality: {
                field: fieldName
              }
            }
          }
        }
      };
      if (subTypeNested) {
        textOrKeywordQuery = {
          nestedSuggestions: {
            nested: {
              path: subTypeNested.nested.path
            },
            aggs: {
              ...textOrKeywordQuery
            }
          }
        };
      }
      return textOrKeywordQuery;
    },
    parse: (rawEsResult, {
      fieldSpec
    }) => {
      var _get;
      let basePath = 'aggregations';
      const isNested = fieldSpec && (0, _common.getFieldSubtypeNested)(fieldSpec);
      basePath += isNested ? '.nestedSuggestions.filteredSuggestions' : '.filteredSuggestions';
      const suggestions = (_get = (0, _lodash.get)(rawEsResult, `${basePath}.suggestions.buckets`)) === null || _get === void 0 ? void 0 : _get.reduce((acc, suggestion) => {
        acc.push({
          value: suggestion.key,
          docCount: suggestion.doc_count
        });
        return acc;
      }, []);
      return {
        suggestions,
        totalCardinality: (0, _lodash.get)(rawEsResult, `${basePath}.unique_terms.value`)
      };
    }
  },
  /**
   * the "IP" query / parser should be used when the options list is built on a field of type IP.
   */
  ip: {
    buildAggregation: ({
      fieldName,
      searchString,
      sort,
      size
    }) => {
      const filteredSuggestions = {
        terms: {
          size,
          field: fieldName,
          shard_size: 10,
          order: (0, _options_list_suggestion_query_helpers.getSortType)(sort)
        }
      };
      const ipRangeQuery = (0, _ip_search.getIpRangeQuery)(searchString !== null && searchString !== void 0 ? searchString : '');
      if (!ipRangeQuery.validSearch) {
        return {};
      }
      return {
        suggestions: {
          ip_range: {
            field: fieldName,
            ranges: ipRangeQuery.rangeQuery,
            keyed: true
          },
          aggs: {
            filteredSuggestions,
            unique_terms: {
              cardinality: {
                field: fieldName
              }
            }
          }
        }
      };
    },
    parse: (rawEsResult, {
      searchString,
      sort,
      fieldSpec,
      size,
      searchTechnique
    }) => {
      var _get2, _get3;
      if (!searchString || !(0, _is_valid_search.isValidSearch)({
        searchString,
        fieldType: fieldSpec === null || fieldSpec === void 0 ? void 0 : fieldSpec.type,
        searchTechnique
      })) {
        // if this is happens, that means there is an invalid search that snuck through to the server side code;
        // so, might as well early return with no suggestions
        return {
          suggestions: [],
          totalCardinality: 0
        };
      }
      const buckets = [];
      (0, _options_list_suggestion_query_helpers.getIpBuckets)(rawEsResult, buckets, 'ipv4'); // modifies buckets array directly, i.e. "by reference"
      (0, _options_list_suggestion_query_helpers.getIpBuckets)(rawEsResult, buckets, 'ipv6');
      const sortedSuggestions = (sort === null || sort === void 0 ? void 0 : sort.direction) === 'asc' ? buckets.sort((bucketA, bucketB) => bucketA.doc_count - bucketB.doc_count) : buckets.sort((bucketA, bucketB) => bucketB.doc_count - bucketA.doc_count);
      const suggestions = sortedSuggestions.slice(0, size).reduce((acc, suggestion) => {
        acc.push({
          value: suggestion.key,
          docCount: suggestion.doc_count
        });
        return acc;
      }, []);
      const totalCardinality = ((_get2 = (0, _lodash.get)(rawEsResult, `aggregations.suggestions.buckets.ipv4.unique_terms.value`)) !== null && _get2 !== void 0 ? _get2 : 0) + ((_get3 = (0, _lodash.get)(rawEsResult, `aggregations.suggestions.buckets.ipv6.unique_terms.value`)) !== null && _get3 !== void 0 ? _get3 : 0);
      return {
        suggestions,
        totalCardinality
      };
    }
  }
};