"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.SenseEditor = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _lodash = _interopRequireDefault(require("lodash"));
var _hjson = require("hjson");
var _public = require("@kbn/es-ui-shared-plugin/public");
var _row_parser = _interopRequireDefault(require("../../../lib/row_parser"));
var utils = _interopRequireWildcard(require("../../../lib/utils"));
var _es = require("../../../lib/es/es");
var _factories = require("../../factories");
var _autocomplete = _interopRequireDefault(require("../../../lib/autocomplete/autocomplete"));
var _services = require("../../../services");
var _constants = require("../../../../common/constants");
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

const {
  collapseLiteralStrings
} = _public.XJson;
class SenseEditor {
  constructor(coreEditor) {
    (0, _defineProperty2.default)(this, "currentReqRange", void 0);
    (0, _defineProperty2.default)(this, "parser", void 0);
    (0, _defineProperty2.default)(this, "autocomplete", void 0);
    (0, _defineProperty2.default)(this, "prevRequestStart", rowOrPos => {
      let curRow;
      if (rowOrPos == null) {
        curRow = this.coreEditor.getCurrentPosition().lineNumber;
      } else if (_lodash.default.isObject(rowOrPos)) {
        curRow = rowOrPos.lineNumber;
      } else {
        curRow = rowOrPos;
      }
      while (curRow > 0 && !this.parser.isStartRequestRow(curRow, this.coreEditor)) curRow--;
      return {
        lineNumber: curRow,
        column: 1
      };
    });
    (0, _defineProperty2.default)(this, "nextRequestStart", rowOrPos => {
      let curRow;
      if (rowOrPos == null) {
        curRow = this.coreEditor.getCurrentPosition().lineNumber;
      } else if (_lodash.default.isObject(rowOrPos)) {
        curRow = rowOrPos.lineNumber;
      } else {
        curRow = rowOrPos;
      }
      const maxLines = this.coreEditor.getLineCount();
      for (; curRow < maxLines - 1; curRow++) {
        if (this.parser.isStartRequestRow(curRow, this.coreEditor)) {
          break;
        }
      }
      return {
        row: curRow,
        column: 0
      };
    });
    (0, _defineProperty2.default)(this, "autoIndent", _lodash.default.debounce(async () => {
      await this.coreEditor.waitForLatestTokens();
      const reqRange = await this.getRequestRange();
      if (!reqRange) {
        return;
      }
      const parsedReq = await this.getRequest();
      if (!parsedReq) {
        return;
      }
      if (parsedReq.data.some(doc => utils.hasComments(doc))) {
        /**
         * Comments require different approach for indentation and do not have condensed format
         * We need to delegate indentation logic to coreEditor since it has access to session and other methods used for formatting and indenting the comments
         */
        this.coreEditor.autoIndent(parsedReq.range);
        return;
      }
      if (parsedReq.data && parsedReq.data.length > 0) {
        let indent = parsedReq.data.length === 1; // unindent multi docs by default
        let formattedData = utils.formatRequestBodyDoc(parsedReq.data, indent);
        if (!formattedData.changed) {
          // toggle.
          indent = !indent;
          formattedData = utils.formatRequestBodyDoc(parsedReq.data, indent);
        }
        parsedReq.data = formattedData.data;
        this.replaceRequestRange(parsedReq, reqRange);
      }
    }, 25));
    (0, _defineProperty2.default)(this, "update", async (data, reTokenizeAll = false) => {
      return this.coreEditor.setValue(data, reTokenizeAll);
    });
    (0, _defineProperty2.default)(this, "replaceRequestRange", (newRequest, requestRange) => {
      const text = utils.textFromRequest(newRequest);
      if (requestRange) {
        this.coreEditor.replaceRange(requestRange, text);
      } else {
        // just insert where we are
        this.coreEditor.insert(this.coreEditor.getCurrentPosition(), text);
      }
    });
    (0, _defineProperty2.default)(this, "getRequestRange", async lineNumber => {
      await this.coreEditor.waitForLatestTokens();
      if (this.parser.isInBetweenRequestsRow(lineNumber)) {
        return null;
      }
      const reqStart = this.prevRequestStart(lineNumber);
      const reqEnd = this.nextRequestEnd(reqStart);
      return {
        start: {
          ...reqStart
        },
        end: {
          ...reqEnd
        }
      };
    });
    (0, _defineProperty2.default)(this, "expandRangeToRequestEdges", async (range = this.coreEditor.getSelectionRange()) => {
      await this.coreEditor.waitForLatestTokens();
      let startLineNumber = range.start.lineNumber;
      let endLineNumber = range.end.lineNumber;
      const maxLine = Math.max(1, this.coreEditor.getLineCount());
      if (this.parser.isInBetweenRequestsRow(startLineNumber)) {
        /* Do nothing... */
      } else {
        for (; startLineNumber >= 1; startLineNumber--) {
          if (this.parser.isStartRequestRow(startLineNumber)) {
            break;
          }
        }
      }
      if (startLineNumber < 1 || startLineNumber > endLineNumber) {
        return null;
      }
      // move end row to the previous request end if between requests, otherwise walk forward
      if (this.parser.isInBetweenRequestsRow(endLineNumber)) {
        for (; endLineNumber >= startLineNumber; endLineNumber--) {
          if (this.parser.isEndRequestRow(endLineNumber)) {
            break;
          }
        }
      } else {
        for (; endLineNumber <= maxLine; endLineNumber++) {
          if (this.parser.isEndRequestRow(endLineNumber)) {
            break;
          }
        }
      }
      if (endLineNumber < startLineNumber || endLineNumber > maxLine) {
        return null;
      }
      const endColumn = (this.coreEditor.getLineValue(endLineNumber) || '').replace(/\s+$/, '').length + 1;
      return {
        start: {
          lineNumber: startLineNumber,
          column: 1
        },
        end: {
          lineNumber: endLineNumber,
          column: endColumn
        }
      };
    });
    (0, _defineProperty2.default)(this, "getRequestInRange", async range => {
      await this.coreEditor.waitForLatestTokens();
      if (!range) {
        return null;
      }
      const request = {
        method: '',
        data: [],
        url: '',
        range
      };
      const pos = range.start;
      const tokenIter = (0, _factories.createTokenIterator)({
        editor: this.coreEditor,
        position: pos
      });
      let t = tokenIter.getCurrentToken();
      if (this.parser.isEmptyToken(t)) {
        // if the row starts with some spaces, skip them.
        t = this.parser.nextNonEmptyToken(tokenIter);
      }
      if (t == null) {
        return null;
      }
      request.method = t.value;
      t = this.parser.nextNonEmptyToken(tokenIter);
      if (!t || t.type === 'method') {
        return null;
      }
      request.url = '';
      while (t && t.type && (t.type.indexOf('url') === 0 || t.type === 'variable.template')) {
        request.url += t.value;
        t = tokenIter.stepForward();
      }
      if (this.parser.isEmptyToken(t)) {
        // if the url row ends with some spaces, skip them.
        t = this.parser.nextNonEmptyToken(tokenIter);
      }

      // If the url row ends with a comment, skip it
      while (this.parser.isCommentToken(t)) {
        t = tokenIter.stepForward();
      }
      let bodyStartLineNumber = (t ? 0 : 1) + tokenIter.getCurrentPosition().lineNumber; // artificially increase end of docs.
      let dataEndPos;
      while (bodyStartLineNumber < range.end.lineNumber || bodyStartLineNumber === range.end.lineNumber && 1 < range.end.column) {
        dataEndPos = this.nextDataDocEnd({
          lineNumber: bodyStartLineNumber,
          column: 1
        });
        const bodyRange = {
          start: {
            lineNumber: bodyStartLineNumber,
            column: 1
          },
          end: dataEndPos
        };
        const data = this.coreEditor.getValueInRange(bodyRange);
        request.data.push(data.trim());
        bodyStartLineNumber = dataEndPos.lineNumber + 1;
      }
      return request;
    });
    (0, _defineProperty2.default)(this, "getRequestsInRange", async (range = this.coreEditor.getSelectionRange(), includeNonRequestBlocks = false) => {
      await this.coreEditor.waitForLatestTokens();
      if (!range) {
        return [];
      }
      const expandedRange = await this.expandRangeToRequestEdges(range);
      if (!expandedRange) {
        return [];
      }
      const requests = [];
      let rangeStartCursor = expandedRange.start.lineNumber;
      const endLineNumber = expandedRange.end.lineNumber;

      // move to the next request start (during the second iterations this may not be exactly on a request
      let currentLineNumber = expandedRange.start.lineNumber;
      const flushNonRequestBlock = () => {
        if (includeNonRequestBlocks) {
          const nonRequestPrefixBlock = this.coreEditor.getLines(rangeStartCursor, currentLineNumber - 1).join('\n');
          if (nonRequestPrefixBlock) {
            requests.push(nonRequestPrefixBlock);
          }
        }
      };
      while (currentLineNumber <= endLineNumber) {
        if (this.parser.isStartRequestRow(currentLineNumber)) {
          flushNonRequestBlock();
          const request = await this.getRequest(currentLineNumber);
          if (!request) {
            // Something has probably gone wrong.
            return requests;
          } else {
            requests.push(request);
            rangeStartCursor = currentLineNumber = request.range.end.lineNumber + 1;
          }
        } else {
          ++currentLineNumber;
        }
      }
      flushNonRequestBlock();
      return requests;
    });
    (0, _defineProperty2.default)(this, "getRequest", async row => {
      await this.coreEditor.waitForLatestTokens();
      if (this.parser.isInBetweenRequestsRow(row)) {
        return null;
      }
      const range = await this.getRequestRange(row);
      return this.getRequestInRange(range);
    });
    (0, _defineProperty2.default)(this, "moveToPreviousRequestEdge", async () => {
      await this.coreEditor.waitForLatestTokens();
      const pos = this.coreEditor.getCurrentPosition();
      for (pos.lineNumber--; pos.lineNumber > 1 && !this.parser.isRequestEdge(pos.lineNumber); pos.lineNumber--) {
        // loop for side effects
      }
      this.coreEditor.moveCursorToPosition({
        lineNumber: pos.lineNumber,
        column: 1
      });
    });
    (0, _defineProperty2.default)(this, "moveToNextRequestEdge", async moveOnlyIfNotOnEdge => {
      await this.coreEditor.waitForLatestTokens();
      const pos = this.coreEditor.getCurrentPosition();
      const maxRow = this.coreEditor.getLineCount();
      if (!moveOnlyIfNotOnEdge) {
        pos.lineNumber++;
      }
      for (; pos.lineNumber < maxRow && !this.parser.isRequestEdge(pos.lineNumber); pos.lineNumber++) {
        // loop for side effects
      }
      this.coreEditor.moveCursorToPosition({
        lineNumber: pos.lineNumber,
        column: 1
      });
    });
    (0, _defineProperty2.default)(this, "nextRequestEnd", pos => {
      pos = pos || this.coreEditor.getCurrentPosition();
      const maxLines = this.coreEditor.getLineCount();
      let curLineNumber = pos.lineNumber;
      for (; curLineNumber <= maxLines; ++curLineNumber) {
        const curRowMode = this.parser.getRowParseMode(curLineNumber);
        // eslint-disable-next-line no-bitwise
        if ((curRowMode & this.parser.MODE.REQUEST_END) > 0) {
          break;
        }
        // eslint-disable-next-line no-bitwise
        if (curLineNumber !== pos.lineNumber && (curRowMode & this.parser.MODE.REQUEST_START) > 0) {
          break;
        }
      }
      const column = (this.coreEditor.getLineValue(curLineNumber) || '').replace(/\s+$/, '').length + 1;
      return {
        lineNumber: curLineNumber,
        column
      };
    });
    (0, _defineProperty2.default)(this, "nextDataDocEnd", pos => {
      pos = pos || this.coreEditor.getCurrentPosition();
      let curLineNumber = pos.lineNumber;
      const maxLines = this.coreEditor.getLineCount();
      for (; curLineNumber < maxLines; curLineNumber++) {
        const curRowMode = this.parser.getRowParseMode(curLineNumber);
        // eslint-disable-next-line no-bitwise
        if ((curRowMode & this.parser.MODE.REQUEST_END) > 0) {
          break;
        }
        // eslint-disable-next-line no-bitwise
        if ((curRowMode & this.parser.MODE.MULTI_DOC_CUR_DOC_END) > 0) {
          break;
        }
        // eslint-disable-next-line no-bitwise
        if (curLineNumber !== pos.lineNumber && (curRowMode & this.parser.MODE.REQUEST_START) > 0) {
          break;
        }
      }
      const column = (this.coreEditor.getLineValue(curLineNumber) || '').length + 1; /* Range goes to 1 after last char */

      return {
        lineNumber: curLineNumber,
        column
      };
    });
    (0, _defineProperty2.default)(this, "highlightCurrentRequestsAndUpdateActionBar", _lodash.default.debounce(async () => {
      await this.coreEditor.waitForLatestTokens();
      const expandedRange = await this.expandRangeToRequestEdges();
      if (expandedRange === null && this.currentReqRange === null) {
        return;
      }
      if (expandedRange !== null && this.currentReqRange !== null && expandedRange.start.lineNumber === this.currentReqRange.start.lineNumber && expandedRange.end.lineNumber === this.currentReqRange.end.lineNumber) {
        // same request, now see if we are on the first line and update the action bar
        const cursorLineNumber = this.coreEditor.getCurrentPosition().lineNumber;
        if (cursorLineNumber === this.currentReqRange.start.lineNumber) {
          this.updateActionsBar();
        }
        return; // nothing to do..
      }
      if (this.currentReqRange) {
        this.coreEditor.removeMarker(this.currentReqRange.markerRef);
      }
      this.currentReqRange = expandedRange;
      if (this.currentReqRange) {
        this.currentReqRange.markerRef = this.coreEditor.addMarker(this.currentReqRange);
      }
      this.updateActionsBar();
    }, 25));
    (0, _defineProperty2.default)(this, "getRequestsAsCURL", async (elasticsearchBaseUrl, range) => {
      const variables = (0, _services.getStorage)().get(_services.StorageKeys.VARIABLES, _constants.DEFAULT_VARIABLES);
      let requests = await this.getRequestsInRange(range, true);
      requests = utils.replaceVariables(requests, variables);
      const result = _lodash.default.map(requests, req => {
        if (typeof req === 'string') {
          // no request block
          return req;
        }
        const path = req.url;
        const method = req.method;
        const data = req.data;

        // this is the first url defined in elasticsearch.hosts
        const url = (0, _es.constructUrl)(elasticsearchBaseUrl, path);

        // Append 'kbn-xsrf' header to bypass (XSRF/CSRF) protections
        let ret = `curl -X${method.toUpperCase()} "${url}" -H "kbn-xsrf: reporting"`;
        if (data && data.length) {
          const joinedData = data.join('\n');
          let dataAsString;
          try {
            ret += ` -H "Content-Type: application/json" -d'\n`;
            if (utils.hasComments(joinedData)) {
              // if there are comments in the data, we need to strip them out
              const dataWithoutComments = (0, _hjson.parse)(joinedData);
              dataAsString = collapseLiteralStrings(JSON.stringify(dataWithoutComments, null, 2));
            } else {
              dataAsString = collapseLiteralStrings(joinedData);
            }
            // We escape single quoted strings that are wrapped in single quoted strings
            ret += dataAsString.replace(/'/g, "'\\''");
            if (data.length > 1) {
              ret += '\n';
            } // end with a new line
            ret += "'";
          } catch (e) {
            throw new Error(`Error parsing data: ${e.message}`);
          }
        }
        return ret;
      });
      return result.join('\n');
    });
    (0, _defineProperty2.default)(this, "updateActionsBar", () => {
      return this.coreEditor.legacyUpdateUI(this.currentReqRange);
    });
    this.coreEditor = coreEditor;
    this.currentReqRange = null;
    this.parser = new _row_parser.default(this.coreEditor);
    this.autocomplete = (0, _autocomplete.default)({
      coreEditor,
      parser: this.parser
    });
    this.coreEditor.registerAutocompleter(this.autocomplete.getCompletions);
    this.coreEditor.on('tokenizerUpdate', this.highlightCurrentRequestsAndUpdateActionBar.bind(this));
    this.coreEditor.on('changeCursor', this.highlightCurrentRequestsAndUpdateActionBar.bind(this));
    this.coreEditor.on('changeScrollTop', this.updateActionsBar.bind(this));
  }
  getCoreEditor() {
    return this.coreEditor;
  }
}
exports.SenseEditor = SenseEditor;