"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateCustomFieldKeysAgainstConfiguration = void 0;
exports.validateCustomFieldTypesInRequest = validateCustomFieldTypesInRequest;
exports.validateSearchCasesCustomFields = exports.validateRequiredCustomFields = exports.validateCustomFields = void 0;
var _lodash = require("lodash");
var _boom = _interopRequireDefault(require("@hapi/boom"));
var _validators = require("../validators");
var _custom_fields = require("../../custom_fields");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const validateCustomFields = params => {
  (0, _validators.validateDuplicatedCustomFieldKeysInRequest)(params);
  validateCustomFieldKeysAgainstConfiguration(params);
  validateRequiredCustomFields(params);
  validateCustomFieldTypesInRequest(params);
};

/**
 * Throws if the type doesn't match the configuration.
 */
exports.validateCustomFields = validateCustomFields;
function validateCustomFieldTypesInRequest({
  requestCustomFields,
  customFieldsConfiguration
}) {
  if (!Array.isArray(requestCustomFields) || !requestCustomFields.length) {
    return;
  }
  if (customFieldsConfiguration === undefined) {
    throw _boom.default.badRequest('No custom fields configured.');
  }
  const invalidCustomFields = (0, _lodash.intersectionWith)(customFieldsConfiguration, requestCustomFields, (requiredVal, requestedVal) => requiredVal.key === requestedVal.key && requiredVal.type !== requestedVal.type).map(config => `"${config.label ? config.label : 'Unknown'}"`);
  if (invalidCustomFields.length) {
    throw _boom.default.badRequest(`The following custom fields have the wrong type in the request: ${invalidCustomFields.join(', ')}`);
  }
}

/**
 * Throws if the key doesn't match the configuration or is missing
 */
const validateCustomFieldKeysAgainstConfiguration = ({
  requestCustomFields,
  customFieldsConfiguration
}) => {
  if (!Array.isArray(requestCustomFields) || !requestCustomFields.length) {
    return [];
  }
  if (customFieldsConfiguration === undefined) {
    throw _boom.default.badRequest('No custom fields configured.');
  }
  const invalidCustomFieldKeys = (0, _lodash.differenceWith)(requestCustomFields, customFieldsConfiguration, (requestVal, configurationVal) => requestVal.key === configurationVal.key).map(e => e.key);
  if (invalidCustomFieldKeys.length) {
    throw _boom.default.badRequest(`Invalid custom field keys: ${invalidCustomFieldKeys}`);
  }
};

/**
 * Returns a list of required custom fields missing from the request
 * that don't have a default value configured.
 */
exports.validateCustomFieldKeysAgainstConfiguration = validateCustomFieldKeysAgainstConfiguration;
const validateRequiredCustomFields = ({
  requestCustomFields,
  customFieldsConfiguration
}) => {
  if (customFieldsConfiguration === undefined) {
    if (!Array.isArray(requestCustomFields) || !requestCustomFields.length) {
      return;
    } else {
      throw _boom.default.badRequest('No custom fields configured.');
    }
  }
  const requiredCustomFields = customFieldsConfiguration.filter(customField => customField.required);
  if (!requiredCustomFields.length) {
    return;
  }
  const missingRequiredCustomFields = (0, _lodash.differenceWith)(requiredCustomFields, requestCustomFields !== null && requestCustomFields !== void 0 ? requestCustomFields : [], (configuration, request) => configuration.key === request.key) // missing custom field and missing defaultValue -> error
  .filter(customField => customField.defaultValue === undefined || customField.defaultValue === null).map(e => `"${e.label}"`);
  if (missingRequiredCustomFields.length) {
    throw _boom.default.badRequest(`Missing required custom fields without default value configured: ${missingRequiredCustomFields.join(', ')}`);
  }
  const nullRequiredCustomFields = requiredCustomFields.filter(requiredField => {
    const found = requestCustomFields === null || requestCustomFields === void 0 ? void 0 : requestCustomFields.find(requestField => requestField.key === requiredField.key);

    // required custom fields cannot be set to null
    return found && found.value === null;
  }).map(e => `"${e.label}"`);
  if (nullRequiredCustomFields.length) {
    throw _boom.default.badRequest(`Invalid value "null" supplied for the following required custom fields: ${nullRequiredCustomFields.join(', ')}`);
  }
};
exports.validateRequiredCustomFields = validateRequiredCustomFields;
const validateSearchCasesCustomFields = ({
  customFieldsConfiguration,
  customFields
}) => {
  let customFieldsMapping = null;
  if (!customFields || (0, _lodash.isEmpty)(customFields)) {
    return;
  }
  if (!customFieldsConfiguration.length) {
    throw _boom.default.badRequest('No custom fields configured.');
  }
  if (Object.keys(customFields).length > _constants.MAX_CUSTOM_FIELDS_PER_CASE) {
    throw _boom.default.badRequest(`Maximum ${_constants.MAX_CUSTOM_FIELDS_PER_CASE} customFields are allowed.`);
  }
  Object.entries(customFields).forEach(([key, value]) => {
    var _customFieldsMapping, _customFieldsMapping2;
    const customFieldConfig = customFieldsConfiguration.find(config => config.key === key);
    if (!customFieldConfig) {
      throw _boom.default.badRequest(`Invalid custom field key: ${key}.`);
    }
    customFieldsMapping = _custom_fields.casesCustomFields.get(customFieldConfig.type);
    if (!((_customFieldsMapping = customFieldsMapping) !== null && _customFieldsMapping !== void 0 && _customFieldsMapping.isFilterable)) {
      throw _boom.default.badRequest(`Filtering by custom field of type ${customFieldConfig.type} is not allowed.`);
    }
    (_customFieldsMapping2 = customFieldsMapping) === null || _customFieldsMapping2 === void 0 ? void 0 : _customFieldsMapping2.validateFilteringValues(value);
  });
};
exports.validateSearchCasesCustomFields = validateSearchCasesCustomFields;