"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTransactionsPerMinute = getTransactionsPerMinute;
var _server = require("@kbn/observability-plugin/server");
var _transaction_types = require("../../../common/transaction_types");
var _apm = require("../../../common/es_fields/apm");
var _transactions = require("../../lib/helpers/transactions");
var _calculate_throughput = require("../../lib/helpers/calculate_throughput");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function getTransactionsPerMinute({
  apmEventClient,
  bucketSize,
  searchAggregatedTransactions,
  start,
  end,
  intervalString
}) {
  const {
    aggregations
  } = await apmEventClient.search('observability_overview_get_transactions_per_minute', {
    apm: {
      events: [(0, _transactions.getProcessorEventForTransactions)(searchAggregatedTransactions)]
    },
    body: {
      track_total_hits: false,
      size: 0,
      query: {
        bool: {
          filter: [...(0, _server.rangeQuery)(start, end), ...(0, _transactions.getBackwardCompatibleDocumentTypeFilter)(searchAggregatedTransactions)]
        }
      },
      aggs: {
        transactionType: {
          terms: {
            field: _apm.TRANSACTION_TYPE
          },
          aggs: {
            timeseries: {
              date_histogram: {
                field: '@timestamp',
                fixed_interval: intervalString,
                min_doc_count: 0
              },
              aggs: {
                throughput: {
                  rate: {
                    unit: 'minute'
                  }
                }
              }
            }
          }
        }
      }
    }
  });
  if (!aggregations || !aggregations.transactionType.buckets) {
    return {
      value: undefined,
      timeseries: []
    };
  }
  const topTransactionTypeBucket = aggregations.transactionType.buckets.find(({
    key: transactionType
  }) => (0, _transaction_types.isDefaultTransactionType)(transactionType)) || aggregations.transactionType.buckets[0];
  return {
    value: (0, _calculate_throughput.calculateThroughputWithRange)({
      start,
      end,
      value: (topTransactionTypeBucket === null || topTransactionTypeBucket === void 0 ? void 0 : topTransactionTypeBucket.doc_count) || 0
    }),
    timeseries: (topTransactionTypeBucket === null || topTransactionTypeBucket === void 0 ? void 0 : topTransactionTypeBucket.timeseries.buckets.map(bucket => ({
      x: bucket.key,
      y: bucket.throughput.value
    }))) || []
  };
}