"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAlertDetailsContextHandler = void 0;
var _moment = _interopRequireDefault(require("moment"));
var _lodash = require("lodash");
var _get_apm_alerts_client = require("../../../lib/helpers/get_apm_alerts_client");
var _get_apm_event_client = require("../../../lib/helpers/get_apm_event_client");
var _get_ml_client = require("../../../lib/helpers/get_ml_client");
var _get_random_sampler = require("../../../lib/helpers/get_random_sampler");
var _get_apm_service_summary = require("../get_apm_service_summary");
var _get_apm_downstream_dependencies = require("../get_apm_downstream_dependencies");
var _get_log_categories = require("../get_log_categories");
var _get_anomalies = require("../get_apm_service_summary/get_anomalies");
var _get_service_name_from_signals = require("./get_service_name_from_signals");
var _get_container_id_from_signals = require("./get_container_id_from_signals");
var _get_changepoints = require("../get_changepoints");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAlertDetailsContextHandler = (resourcePlugins, logger) => {
  return async (requestContext, query) => {
    const resources = {
      getApmIndices: async () => {
        const coreContext = await requestContext.core;
        return resourcePlugins.apmDataAccess.setup.getApmIndices(coreContext.savedObjects.client);
      },
      request: requestContext.request,
      params: {
        query: {
          _inspect: false
        }
      },
      plugins: resourcePlugins,
      context: {
        core: requestContext.core,
        licensing: requestContext.licensing,
        alerting: resourcePlugins.alerting.start().then(startContract => {
          return {
            getRulesClient() {
              return startContract.getRulesClientWithRequest(requestContext.request);
            }
          };
        }),
        rac: resourcePlugins.ruleRegistry.start().then(startContract => {
          return {
            getAlertsClient() {
              return startContract.getRacClientWithRequest(requestContext.request);
            }
          };
        })
      }
    };
    const [apmEventClient, annotationsClient, apmAlertsClient, coreContext, mlClient] = await Promise.all([(0, _get_apm_event_client.getApmEventClient)(resources), resourcePlugins.observability.setup.getScopedAnnotationsClient(resources.context, requestContext.request), (0, _get_apm_alerts_client.getApmAlertsClient)(resources), requestContext.core, (0, _get_ml_client.getMlClient)(resources), (0, _get_random_sampler.getRandomSampler)({
      security: resourcePlugins.security,
      probability: 1,
      request: requestContext.request
    })]);
    const esClient = coreContext.elasticsearch.client.asCurrentUser;
    const alertStartedAt = query.alert_started_at;
    const serviceEnvironment = query['service.environment'];
    const hostName = query['host.name'];
    const kubernetesPodName = query['kubernetes.pod.name'];
    const [serviceName, containerId] = await Promise.all([(0, _get_service_name_from_signals.getServiceNameFromSignals)({
      query,
      esClient,
      coreContext,
      apmEventClient
    }), (0, _get_container_id_from_signals.getContainerIdFromSignals)({
      query,
      esClient,
      coreContext,
      apmEventClient
    })]);
    async function handleError(cb) {
      try {
        return await cb();
      } catch (error) {
        logger.error('Error while fetching observability alert details context');
        logger.error(error);
        return;
      }
    }
    const serviceSummaryPromise = serviceName ? handleError(() => (0, _get_apm_service_summary.getApmServiceSummary)({
      apmEventClient,
      annotationsClient,
      esClient,
      apmAlertsClient,
      mlClient,
      logger,
      arguments: {
        'service.name': serviceName,
        'service.environment': serviceEnvironment,
        start: (0, _moment.default)(alertStartedAt).subtract(5, 'minute').toISOString(),
        end: alertStartedAt
      }
    })) : undefined;
    const downstreamDependenciesPromise = serviceName ? handleError(() => (0, _get_apm_downstream_dependencies.getAssistantDownstreamDependencies)({
      apmEventClient,
      arguments: {
        'service.name': serviceName,
        'service.environment': serviceEnvironment,
        start: (0, _moment.default)(alertStartedAt).subtract(15, 'minute').toISOString(),
        end: alertStartedAt
      }
    })) : undefined;
    const logCategoriesPromise = handleError(() => (0, _get_log_categories.getLogCategories)({
      esClient,
      coreContext,
      arguments: {
        start: (0, _moment.default)(alertStartedAt).subtract(15, 'minute').toISOString(),
        end: alertStartedAt,
        'service.name': serviceName,
        'host.name': hostName,
        'container.id': containerId,
        'kubernetes.pod.name': kubernetesPodName
      }
    }));
    const serviceChangePointsPromise = handleError(() => (0, _get_changepoints.getServiceChangePoints)({
      apmEventClient,
      start: (0, _moment.default)(alertStartedAt).subtract(6, 'hours').toISOString(),
      end: alertStartedAt,
      serviceName,
      serviceEnvironment,
      transactionType: query['transaction.type'],
      transactionName: query['transaction.name']
    }));
    const exitSpanChangePointsPromise = handleError(() => (0, _get_changepoints.getExitSpanChangePoints)({
      apmEventClient,
      start: (0, _moment.default)(alertStartedAt).subtract(6, 'hours').toISOString(),
      end: alertStartedAt,
      serviceName,
      serviceEnvironment
    }));
    const anomaliesPromise = handleError(() => (0, _get_anomalies.getAnomalies)({
      start: (0, _moment.default)(alertStartedAt).subtract(1, 'hour').valueOf(),
      end: (0, _moment.default)(alertStartedAt).valueOf(),
      environment: serviceEnvironment,
      mlClient,
      logger
    }));
    const [serviceSummary, downstreamDependencies, logCategories, serviceChangePoints, exitSpanChangePoints, anomalies] = await Promise.all([serviceSummaryPromise, downstreamDependenciesPromise, logCategoriesPromise, serviceChangePointsPromise, exitSpanChangePointsPromise, anomaliesPromise]);
    return [{
      key: 'serviceSummary',
      description: 'Metadata for the service where the alert occurred',
      data: serviceSummary
    }, {
      key: 'downstreamDependencies',
      description: `Downstream dependencies from the service "${serviceName}". Problems in these services can negatively affect the performance of "${serviceName}"`,
      data: downstreamDependencies
    }, {
      key: 'serviceChangePoints',
      description: `Significant change points for "${serviceName}". Use this to spot dips and spikes in throughput, latency and failure rate`,
      data: serviceChangePoints
    }, {
      key: 'exitSpanChangePoints',
      description: `Significant change points for the dependencies of "${serviceName}". Use this to spot dips or spikes in throughput, latency and failure rate for downstream dependencies`,
      data: exitSpanChangePoints
    }, {
      key: 'logCategories',
      description: `Log events occurring around the time of the alert`,
      data: logCategories
    }, {
      key: 'anomalies',
      description: `Anomalies for services running in the environment "${serviceEnvironment}"`,
      data: anomalies
    }].filter(({
      data
    }) => !(0, _lodash.isEmpty)(data));
  };
};
exports.getAlertDetailsContextHandler = getAlertDetailsContextHandler;