"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WaterfallLegendType = void 0;
exports.getClockSkew = getClockSkew;
exports.getOrderedWaterfallItems = getOrderedWaterfallItems;
exports.getOrphanTraceItemsCount = void 0;
exports.getWaterfall = getWaterfall;
var _eui = require("@elastic/eui");
var _lodash = require("lodash");
var _common = require("@kbn/observability-plugin/common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ROOT_ID = 'root';
let WaterfallLegendType = exports.WaterfallLegendType = /*#__PURE__*/function (WaterfallLegendType) {
  WaterfallLegendType["ServiceName"] = "serviceName";
  WaterfallLegendType["SpanType"] = "spanType";
  return WaterfallLegendType;
}({});
function getLegendValues(transactionOrSpan) {
  return {
    [WaterfallLegendType.ServiceName]: transactionOrSpan.service.name,
    [WaterfallLegendType.SpanType]: transactionOrSpan.processor.event === _common.ProcessorEvent.span ? transactionOrSpan.span.subtype || transactionOrSpan.span.type : ''
  };
}
function getTransactionItem(transaction, linkedChildrenCount = 0) {
  var _transaction$parent, _transaction$span$lin, _transaction$span, _transaction$span$lin2;
  return {
    docType: 'transaction',
    doc: transaction,
    id: transaction.transaction.id,
    parentId: (_transaction$parent = transaction.parent) === null || _transaction$parent === void 0 ? void 0 : _transaction$parent.id,
    duration: transaction.transaction.duration.us,
    offset: 0,
    skew: 0,
    legendValues: getLegendValues(transaction),
    color: '',
    spanLinksCount: {
      linkedParents: (_transaction$span$lin = (_transaction$span = transaction.span) === null || _transaction$span === void 0 ? void 0 : (_transaction$span$lin2 = _transaction$span.links) === null || _transaction$span$lin2 === void 0 ? void 0 : _transaction$span$lin2.length) !== null && _transaction$span$lin !== void 0 ? _transaction$span$lin : 0,
      linkedChildren: linkedChildrenCount
    }
  };
}
function getSpanItem(span, linkedChildrenCount = 0) {
  var _span$parent, _span$span$links$leng, _span$span$links;
  return {
    docType: 'span',
    doc: span,
    id: span.span.id,
    parentId: (_span$parent = span.parent) === null || _span$parent === void 0 ? void 0 : _span$parent.id,
    duration: span.span.duration.us,
    offset: 0,
    skew: 0,
    legendValues: getLegendValues(span),
    color: '',
    spanLinksCount: {
      linkedParents: (_span$span$links$leng = (_span$span$links = span.span.links) === null || _span$span$links === void 0 ? void 0 : _span$span$links.length) !== null && _span$span$links$leng !== void 0 ? _span$span$links$leng : 0,
      linkedChildren: linkedChildrenCount
    }
  };
}
function getErrorItem(error, items, entryWaterfallTransaction) {
  var _entryWaterfallTransa;
  const entryTimestamp = (_entryWaterfallTransa = entryWaterfallTransaction === null || entryWaterfallTransaction === void 0 ? void 0 : entryWaterfallTransaction.doc.timestamp.us) !== null && _entryWaterfallTransa !== void 0 ? _entryWaterfallTransa : 0;
  const parent = items.find(waterfallItem => {
    var _error$parent;
    return waterfallItem.id === ((_error$parent = error.parent) === null || _error$parent === void 0 ? void 0 : _error$parent.id);
  });
  const errorItem = {
    docType: 'error',
    doc: error,
    id: error.error.id,
    parent,
    parentId: parent === null || parent === void 0 ? void 0 : parent.id,
    offset: error.timestamp.us - entryTimestamp,
    skew: 0,
    color: ''
  };
  return {
    ...errorItem,
    skew: getClockSkew(errorItem, parent)
  };
}
function getClockSkew(item, parentItem) {
  if (!parentItem) {
    return 0;
  }
  switch (item.docType) {
    // don't calculate skew for spans and errors. Just use parent's skew
    case 'error':
    case 'span':
      return parentItem.skew;
    // transaction is the initial entry in a service. Calculate skew for this, and it will be propagated to all child spans
    case 'transaction':
      {
        const parentStart = parentItem.doc.timestamp.us + parentItem.skew;

        // determine if child starts before the parent
        const offsetStart = parentStart - item.doc.timestamp.us;
        if (offsetStart > 0) {
          const latency = Math.max(parentItem.duration - item.duration, 0) / 2;
          return offsetStart + latency;
        }

        // child transaction starts after parent thus no adjustment is needed
        return 0;
      }
  }
}
function getOrderedWaterfallItems(childrenByParentId, entryWaterfallTransaction) {
  if (!entryWaterfallTransaction) {
    return [];
  }
  const entryTimestamp = entryWaterfallTransaction.doc.timestamp.us;
  const visitedWaterfallItemSet = new Set();
  function getSortedChildren(item, parentItem) {
    if (visitedWaterfallItemSet.has(item)) {
      return [];
    }
    visitedWaterfallItemSet.add(item);
    const children = (0, _lodash.sortBy)(childrenByParentId[item.id] || [], 'doc.timestamp.us');
    item.parent = parentItem;
    // get offset from the beginning of trace
    item.offset = item.doc.timestamp.us - entryTimestamp;
    // move the item to the right if it starts before its parent
    item.skew = getClockSkew(item, parentItem);
    const deepChildren = (0, _lodash.flatten)(children.map(child => getSortedChildren(child, item)));
    return [item, ...deepChildren];
  }
  return getSortedChildren(entryWaterfallTransaction);
}
function getRootWaterfallTransaction(childrenByParentId) {
  const item = (0, _lodash.first)(childrenByParentId.root);
  if (item && item.docType === 'transaction') {
    return item;
  }
}
function getLegends(waterfallItems) {
  const onlyBaseSpanItems = waterfallItems.filter(item => item.docType === 'span' || item.docType === 'transaction');
  const legends = [WaterfallLegendType.ServiceName, WaterfallLegendType.SpanType].flatMap(legendType => {
    const allLegendValues = (0, _lodash.uniq)(onlyBaseSpanItems.map(item => item.legendValues[legendType]));
    const palette = (0, _eui.euiPaletteColorBlind)({
      rotations: Math.ceil(allLegendValues.length / 10)
    });
    return allLegendValues.map((value, index) => ({
      type: legendType,
      value,
      color: palette[index]
    }));
  });
  return legends;
}
const getWaterfallDuration = waterfallItems => Math.max(...waterfallItems.map(item => item.offset + item.skew + ('duration' in item ? item.duration : 0)), 0);
const getWaterfallItems = (items, spanLinksCountById) => items.map(item => {
  const docType = item.processor.event;
  switch (docType) {
    case 'span':
      {
        const span = item;
        return getSpanItem(span, spanLinksCountById[span.span.id]);
      }
    case 'transaction':
      const transaction = item;
      return getTransactionItem(transaction, spanLinksCountById[transaction.transaction.id]);
  }
});
function reparentSpans(waterfallItems) {
  // find children that needs to be re-parented and map them to their correct parent id
  const childIdToParentIdMapping = Object.fromEntries((0, _lodash.flatten)(waterfallItems.map(waterfallItem => {
    if (waterfallItem.docType === 'span') {
      var _waterfallItem$doc$ch, _waterfallItem$doc$ch2;
      const childIds = (_waterfallItem$doc$ch = (_waterfallItem$doc$ch2 = waterfallItem.doc.child) === null || _waterfallItem$doc$ch2 === void 0 ? void 0 : _waterfallItem$doc$ch2.id) !== null && _waterfallItem$doc$ch !== void 0 ? _waterfallItem$doc$ch : [];
      return childIds.map(id => [id, waterfallItem.id]);
    }
    return [];
  })));

  // update parent id for children that needs it or return unchanged
  return waterfallItems.map(waterfallItem => {
    const newParentId = childIdToParentIdMapping[waterfallItem.id];
    if (newParentId) {
      return {
        ...waterfallItem,
        parentId: newParentId
      };
    }
    return waterfallItem;
  });
}
const getChildrenGroupedByParentId = waterfallItems => (0, _lodash.groupBy)(waterfallItems, item => item.parentId ? item.parentId : ROOT_ID);
const getEntryWaterfallTransaction = (entryTransactionId, waterfallItems) => waterfallItems.find(item => item.docType === 'transaction' && item.id === entryTransactionId);
function isInEntryTransaction(parentIdLookup, entryTransactionId, currentId) {
  if (currentId === entryTransactionId) {
    return true;
  }
  const parentId = parentIdLookup.get(currentId);
  if (parentId) {
    return isInEntryTransaction(parentIdLookup, entryTransactionId, parentId);
  }
  return false;
}
function getWaterfallErrors(errorDocs, items, entryWaterfallTransaction) {
  const errorItems = errorDocs.map(errorDoc => getErrorItem(errorDoc, items, entryWaterfallTransaction));
  if (!entryWaterfallTransaction) {
    return errorItems;
  }
  const parentIdLookup = [...items, ...errorItems].reduce((map, {
    id,
    parentId
  }) => {
    map.set(id, parentId !== null && parentId !== void 0 ? parentId : ROOT_ID);
    return map;
  }, new Map());
  return errorItems.filter(errorItem => isInEntryTransaction(parentIdLookup, entryWaterfallTransaction === null || entryWaterfallTransaction === void 0 ? void 0 : entryWaterfallTransaction.id, errorItem.id));
}

// map parent.id to the number of errors
/*
  { 'parentId': 2 }
  */
function getErrorCountByParentId(errorDocs) {
  return errorDocs.reduce((acc, doc) => {
    var _doc$parent, _acc$parentId;
    const parentId = (_doc$parent = doc.parent) === null || _doc$parent === void 0 ? void 0 : _doc$parent.id;
    if (!parentId) {
      return acc;
    }
    acc[parentId] = ((_acc$parentId = acc[parentId]) !== null && _acc$parentId !== void 0 ? _acc$parentId : 0) + 1;
    return acc;
  }, {});
}
const getOrphanTraceItemsCount = traceDocs => {
  const waterfallItemsIds = new Set(traceDocs.map(doc => {
    var _doc$transaction;
    return doc.processor.event === 'span' ? (doc === null || doc === void 0 ? void 0 : doc.span).id : doc === null || doc === void 0 ? void 0 : (_doc$transaction = doc.transaction) === null || _doc$transaction === void 0 ? void 0 : _doc$transaction.id;
  }));
  let missingTraceItemsCounter = 0;
  traceDocs.some(item => {
    var _item$parent;
    if ((_item$parent = item.parent) !== null && _item$parent !== void 0 && _item$parent.id && !waterfallItemsIds.has(item.parent.id)) {
      missingTraceItemsCounter++;
    }
  });
  return missingTraceItemsCounter;
};
exports.getOrphanTraceItemsCount = getOrphanTraceItemsCount;
function getWaterfall(apiResponse) {
  const {
    traceItems,
    entryTransaction
  } = apiResponse;
  if ((0, _lodash.isEmpty)(traceItems.traceDocs) || !entryTransaction) {
    return {
      duration: 0,
      items: [],
      legends: [],
      errorItems: [],
      childrenByParentId: {},
      getErrorCount: () => 0,
      exceedsMax: false,
      totalErrorsCount: 0,
      traceDocsTotal: 0,
      maxTraceItems: 0,
      orphanTraceItemsCount: 0
    };
  }
  const errorCountByParentId = getErrorCountByParentId(traceItems.errorDocs);
  const waterfallItems = getWaterfallItems(traceItems.traceDocs, traceItems.spanLinksCountById);
  const childrenByParentId = getChildrenGroupedByParentId(reparentSpans(waterfallItems));
  const entryWaterfallTransaction = getEntryWaterfallTransaction(entryTransaction.transaction.id, waterfallItems);
  const items = getOrderedWaterfallItems(childrenByParentId, entryWaterfallTransaction);
  const errorItems = getWaterfallErrors(traceItems.errorDocs, items, entryWaterfallTransaction);
  const rootWaterfallTransaction = getRootWaterfallTransaction(childrenByParentId);
  const duration = getWaterfallDuration(items);
  const legends = getLegends(items);
  const orphanTraceItemsCount = getOrphanTraceItemsCount(traceItems.traceDocs);
  return {
    entryWaterfallTransaction,
    rootWaterfallTransaction,
    entryTransaction,
    duration,
    items,
    legends,
    errorItems,
    childrenByParentId: getChildrenGroupedByParentId(items),
    getErrorCount: parentId => {
      var _errorCountByParentId;
      return (_errorCountByParentId = errorCountByParentId[parentId]) !== null && _errorCountByParentId !== void 0 ? _errorCountByParentId : 0;
    },
    exceedsMax: traceItems.exceedsMax,
    totalErrorsCount: traceItems.errorDocs.length,
    traceDocsTotal: traceItems.traceDocsTotal,
    maxTraceItems: traceItems.maxTraceItems,
    orphanTraceItemsCount
  };
}