"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RuleTypeRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _coreSavedObjectsUtilsServer = require("@kbn/core-saved-objects-utils-server");
var _server = require("@kbn/task-manager-plugin/server");
var _lodash = require("lodash");
var _lib = require("../lib");
var _get_time_range = require("../lib/get_time_range");
var _types = require("../types");
var _task_runner_timer = require("./task_runner_timer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RuleTypeRunner {
  constructor(options) {
    (0, _defineProperty2.default)(this, "cancelled", false);
    this.options = options;
  }
  cancelRun() {
    this.cancelled = true;
  }
  async run({
    context,
    alertsClient,
    executionId,
    executorServices,
    maintenanceWindows = [],
    maintenanceWindowsWithoutScopedQueryIds = [],
    rule,
    startedAt,
    state,
    validatedParams
  }) {
    const {
      alertTypeId: ruleTypeId,
      consumer,
      schedule,
      throttle = null,
      notifyWhen = null,
      name,
      tags,
      createdBy,
      updatedBy,
      createdAt,
      updatedAt,
      enabled,
      actions,
      muteAll,
      revision,
      snoozeSchedule,
      alertDelay
    } = rule;
    const {
      alertTypeState: ruleTypeState = {},
      previousStartedAt
    } = state;
    const {
      updatedRuleTypeState,
      error,
      stackTrace
    } = await this.options.timer.runWithTimer(_task_runner_timer.TaskRunnerTimerSpan.RuleTypeRun, async () => {
      var _executorResult;
      const checkHasReachedAlertLimit = () => {
        const reachedLimit = alertsClient.hasReachedAlertLimit() || false;
        if (reachedLimit) {
          this.options.logger.warn(`rule execution generated greater than ${this.options.context.maxAlerts} alerts: ${context.ruleLogPrefix}`);
          context.ruleRunMetricsStore.setHasReachedAlertLimit(true);
        }
        return reachedLimit;
      };
      let executorResult;
      try {
        var _context$namespace;
        const ctx = {
          type: 'alert',
          name: `execute ${ruleTypeId}`,
          id: context.ruleId,
          description: `execute [${ruleTypeId}] with name [${name}] in [${(_context$namespace = context.namespace) !== null && _context$namespace !== void 0 ? _context$namespace : _coreSavedObjectsUtilsServer.DEFAULT_NAMESPACE_STRING}] namespace`
        };
        executorResult = await this.options.context.executionContext.withContext(ctx, () => this.options.ruleType.executor({
          executionId,
          services: {
            alertFactory: alertsClient.factory(),
            alertsClient: alertsClient.client(),
            dataViews: executorServices.dataViews,
            ruleMonitoringService: executorServices.ruleMonitoringService,
            ruleResultService: executorServices.ruleResultService,
            savedObjectsClient: executorServices.savedObjectsClient,
            scopedClusterClient: executorServices.wrappedScopedClusterClient.client(),
            searchSourceClient: executorServices.wrappedSearchSourceClient.searchSourceClient,
            share: this.options.context.share,
            shouldStopExecution: () => this.cancelled,
            shouldWriteAlerts: () => this.shouldLogAndScheduleActionsForAlerts(),
            uiSettingsClient: executorServices.uiSettingsClient
          },
          params: validatedParams,
          state: ruleTypeState,
          startedAt: startedAt,
          previousStartedAt: previousStartedAt ? new Date(previousStartedAt) : null,
          spaceId: context.spaceId,
          namespace: context.namespace,
          rule: {
            id: context.ruleId,
            name,
            tags,
            consumer,
            producer: this.options.ruleType.producer,
            revision,
            ruleTypeId,
            ruleTypeName: this.options.ruleType.name,
            enabled,
            schedule,
            actions,
            createdBy,
            updatedBy,
            createdAt,
            updatedAt,
            throttle,
            notifyWhen,
            muteAll,
            snoozeSchedule,
            alertDelay
          },
          logger: this.options.logger,
          flappingSettings: context.flappingSettings,
          // passed in so the rule registry knows about maintenance windows
          ...(maintenanceWindowsWithoutScopedQueryIds.length ? {
            maintenanceWindowIds: maintenanceWindowsWithoutScopedQueryIds
          } : {}),
          getTimeRange: timeWindow => (0, _get_time_range.getTimeRange)(this.options.logger, context.queryDelaySettings, timeWindow)
        }));
        // Rule type execution has successfully completed
        // Check that the rule type either never requested the max alerts limit
        // or requested it and then reported back whether it exceeded the limit
        // If neither of these apply, this check will throw an error
        // These errors should show up during rule type development
        alertsClient.checkLimitUsage();
      } catch (err) {
        // Check if this error is due to reaching the alert limit
        if (!checkHasReachedAlertLimit()) {
          context.alertingEventLogger.setExecutionFailed(`rule execution failure: ${context.ruleLogPrefix}`, err.message);
          return {
            error: (0, _server.createTaskRunError)(new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.Execute, err), _server.TaskErrorSource.FRAMEWORK),
            stackTrace: {
              message: err,
              stackTrace: err.stack
            }
          };
        }
      }

      // Check if the rule type has reported that it reached the alert limit
      checkHasReachedAlertLimit();
      context.alertingEventLogger.setExecutionSucceeded(`rule executed: ${context.ruleLogPrefix}`);
      context.ruleRunMetricsStore.setSearchMetrics([executorServices.wrappedScopedClusterClient.getMetrics(), executorServices.wrappedSearchSourceClient.getMetrics()]);
      return {
        updatedRuleTypeState: ((_executorResult = executorResult) === null || _executorResult === void 0 ? void 0 : _executorResult.state) || undefined
      };
    });
    if (error) {
      return {
        state: undefined,
        error,
        stackTrace
      };
    }
    await this.options.timer.runWithTimer(_task_runner_timer.TaskRunnerTimerSpan.ProcessAlerts, async () => {
      var _alertDelay$active;
      alertsClient.processAlerts({
        flappingSettings: context.flappingSettings,
        notifyOnActionGroupChange: notifyWhen === _types.RuleNotifyWhen.CHANGE || (0, _lodash.some)(actions, action => {
          var _action$frequency;
          return ((_action$frequency = action.frequency) === null || _action$frequency === void 0 ? void 0 : _action$frequency.notifyWhen) === _types.RuleNotifyWhen.CHANGE;
        }),
        maintenanceWindowIds: maintenanceWindowsWithoutScopedQueryIds,
        alertDelay: (_alertDelay$active = alertDelay === null || alertDelay === void 0 ? void 0 : alertDelay.active) !== null && _alertDelay$active !== void 0 ? _alertDelay$active : 0,
        ruleRunMetricsStore: context.ruleRunMetricsStore
      });
    });
    await this.options.timer.runWithTimer(_task_runner_timer.TaskRunnerTimerSpan.PersistAlerts, async () => {
      const updateAlertsMaintenanceWindowResult = await alertsClient.persistAlerts(maintenanceWindows);

      // Set the event log MW ids again, this time including the ids that matched alerts with
      // scoped query
      if (updateAlertsMaintenanceWindowResult !== null && updateAlertsMaintenanceWindowResult !== void 0 && updateAlertsMaintenanceWindowResult.maintenanceWindowIds) {
        context.alertingEventLogger.setMaintenanceWindowIds(updateAlertsMaintenanceWindowResult.maintenanceWindowIds);
      }
    });
    alertsClient.logAlerts({
      eventLogger: context.alertingEventLogger,
      ruleRunMetricsStore: context.ruleRunMetricsStore,
      shouldLogAlerts: this.shouldLogAndScheduleActionsForAlerts()
    });
    return {
      state: updatedRuleTypeState
    };
  }
  shouldLogAndScheduleActionsForAlerts() {
    // if execution hasn't been cancelled, return true
    if (!this.cancelled) {
      return true;
    }

    // if execution has been cancelled, return true if EITHER alerting config or rule type indicate to proceed with scheduling actions
    return !this.options.context.cancelAlertsOnRuleTimeout || !this.options.ruleType.cancelAlertsOnRuleTimeout;
  }
}
exports.RuleTypeRunner = RuleTypeRunner;