"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDecryptedRule = getDecryptedRule;
exports.getFakeKibanaRequest = getFakeKibanaRequest;
exports.validateRuleAndCreateFakeRequest = validateRuleAndCreateFakeRequest;
var _server = require("@kbn/spaces-plugin/server");
var _server2 = require("@kbn/core/server");
var _server3 = require("@kbn/task-manager-plugin/server");
var _lib = require("../lib");
var _types = require("../types");
var _common = require("../../common");
var _saved_objects = require("../saved_objects");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * With the decrypted rule saved object
 * - transform from domain model to application model (rule)
 * - create a fakeRequest object using the rule API key
 * - get an instance of the RulesClient using the fakeRequest
 */
function validateRuleAndCreateFakeRequest(params) {
  const {
    context,
    paramValidator,
    ruleData: {
      rawRule,
      references,
      version
    },
    ruleId,
    ruleTypeRegistry,
    spaceId
  } = params;
  const {
    enabled,
    apiKey,
    alertTypeId: ruleTypeId
  } = rawRule;
  if (!enabled) {
    throw (0, _server3.createTaskRunError)(new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.Disabled, new Error(`Rule failed to execute because rule ran after it was disabled.`)), _server3.TaskErrorSource.FRAMEWORK);
  }
  const fakeRequest = getFakeKibanaRequest(context, spaceId, apiKey);
  const rulesClient = context.getRulesClientWithRequest(fakeRequest);
  const rule = rulesClient.getAlertFromRaw({
    id: ruleId,
    ruleTypeId,
    rawRule,
    references,
    includeLegacyId: false,
    omitGeneratedValues: false
  });
  try {
    ruleTypeRegistry.ensureRuleTypeEnabled(rule.alertTypeId);
  } catch (err) {
    throw (0, _server3.createTaskRunError)(new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.License, err), _server3.TaskErrorSource.USER);
  }
  let validatedParams;
  try {
    validatedParams = (0, _lib.validateRuleTypeParams)(rule.params, paramValidator);
  } catch (err) {
    throw (0, _server3.createTaskRunError)(new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.Validate, err), _server3.TaskErrorSource.USER);
  }
  if (rule.monitoring) {
    if (rule.monitoring.run.history.length >= _common.MONITORING_HISTORY_LIMIT) {
      // Remove the first (oldest) record
      rule.monitoring.run.history.shift();
    }
  }
  return {
    apiKey,
    fakeRequest,
    rule,
    rulesClient,
    validatedParams,
    version
  };
}

/**
 * Loads the decrypted rule saved object
 */
async function getDecryptedRule(context, ruleId, spaceId) {
  const namespace = context.spaceIdToNamespace(spaceId);
  let rawRule;
  try {
    rawRule = await context.encryptedSavedObjectsClient.getDecryptedAsInternalUser(_saved_objects.RULE_SAVED_OBJECT_TYPE, ruleId, {
      namespace
    });
  } catch (e) {
    const error = new _lib.ErrorWithReason(_types.RuleExecutionStatusErrorReasons.Decrypt, e);
    if (_server2.SavedObjectsErrorHelpers.isNotFoundError(e)) {
      throw (0, _server3.createTaskRunError)(error, _server3.TaskErrorSource.USER);
    }
    throw (0, _server3.createTaskRunError)(error, _server3.TaskErrorSource.FRAMEWORK);
  }
  return {
    version: rawRule.version,
    rawRule: rawRule.attributes,
    references: rawRule.references
  };
}
function getFakeKibanaRequest(context, spaceId, apiKey) {
  const requestHeaders = {};
  if (apiKey) {
    requestHeaders.authorization = `ApiKey ${apiKey}`;
  }
  const path = (0, _server.addSpaceIdToPath)('/', spaceId);
  const fakeRawRequest = {
    headers: requestHeaders,
    path: '/'
  };
  const fakeRequest = _server2.CoreKibanaRequest.from(fakeRawRequest);
  context.basePathService.set(fakeRequest, path);
  return fakeRequest;
}