"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getScopedQueryHitsWithIds = exports.getQueryByScopedQueries = exports.getMaintenanceWindowAlertsQuery = exports.getLifecycleAlertsQueries = exports.getHitsWithCount = exports.getContinualAlertsQuery = void 0;
var _ruleDataUtils = require("@kbn/rule-data-utils");
var _common = require("../../../common");
var _format_alert = require("./format_alert");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MAX_ALERT_DOCS_TO_RETURN = 100;
var AlertTypes = /*#__PURE__*/function (AlertTypes) {
  AlertTypes[AlertTypes["NEW"] = 0] = "NEW";
  AlertTypes[AlertTypes["ONGOING"] = 1] = "ONGOING";
  AlertTypes[AlertTypes["RECOVERED"] = 2] = "RECOVERED";
  return AlertTypes;
}(AlertTypes || {});
const getLifecycleAlertsQueryByExecutionUuid = ({
  executionUuid,
  ruleId,
  excludedAlertInstanceIds,
  alertsFilter
}) => {
  // lifecycle alerts assign a different action to an alert depending
  // on whether it is new/ongoing/recovered. query for each action in order
  // to get the count of each action type as well as up to the maximum number
  // of each type of alert.
  return [getQueryByExecutionUuid({
    executionUuid,
    ruleId,
    excludedAlertInstanceIds,
    action: 'open',
    alertsFilter
  }), getQueryByExecutionUuid({
    executionUuid,
    ruleId,
    excludedAlertInstanceIds,
    action: 'active',
    alertsFilter
  }), getQueryByExecutionUuid({
    executionUuid,
    ruleId,
    excludedAlertInstanceIds,
    action: 'close',
    alertsFilter
  })];
};
const getLifecycleAlertsQueryByTimeRange = ({
  start,
  end,
  ruleId,
  excludedAlertInstanceIds,
  alertsFilter
}) => {
  return [getQueryByTimeRange({
    start,
    end,
    ruleId,
    excludedAlertInstanceIds,
    type: AlertTypes.NEW,
    alertsFilter
  }), getQueryByTimeRange({
    start,
    end,
    ruleId,
    excludedAlertInstanceIds,
    type: AlertTypes.ONGOING,
    alertsFilter
  }), getQueryByTimeRange({
    start,
    end,
    ruleId,
    excludedAlertInstanceIds,
    type: AlertTypes.RECOVERED,
    alertsFilter
  })];
};
const getQueryByExecutionUuid = ({
  executionUuid,
  ruleId,
  excludedAlertInstanceIds,
  action,
  alertsFilter
}) => {
  const filter = [{
    term: {
      [_ruleDataUtils.ALERT_RULE_EXECUTION_UUID]: executionUuid
    }
  }, {
    term: {
      [_ruleDataUtils.ALERT_RULE_UUID]: ruleId
    }
  }, {
    bool: {
      must_not: {
        exists: {
          field: _ruleDataUtils.ALERT_MAINTENANCE_WINDOW_IDS
        }
      }
    }
  }];
  if (action) {
    filter.push({
      term: {
        [_ruleDataUtils.EVENT_ACTION]: action
      }
    });
  }
  if (excludedAlertInstanceIds.length) {
    filter.push({
      bool: {
        must_not: {
          terms: {
            [_ruleDataUtils.ALERT_INSTANCE_ID]: excludedAlertInstanceIds
          }
        }
      }
    });
  }
  if (alertsFilter) {
    filter.push(...generateAlertsFilterDSL(alertsFilter));
  }
  return {
    size: MAX_ALERT_DOCS_TO_RETURN,
    track_total_hits: true,
    query: {
      bool: {
        filter
      }
    }
  };
};
const getQueryByTimeRange = ({
  start,
  end,
  ruleId,
  excludedAlertInstanceIds,
  type,
  alertsFilter
}) => {
  // base query filters the alert documents for a rule by the given time range
  let filter = [{
    range: {
      [_ruleDataUtils.TIMESTAMP]: {
        gte: start.toISOString(),
        lt: end.toISOString()
      }
    }
  }, {
    term: {
      [_ruleDataUtils.ALERT_RULE_UUID]: ruleId
    }
  }];
  if (excludedAlertInstanceIds.length) {
    filter.push({
      bool: {
        must_not: {
          terms: {
            [_ruleDataUtils.ALERT_INSTANCE_ID]: excludedAlertInstanceIds
          }
        }
      }
    });
  }
  if (type === AlertTypes.NEW) {
    // alerts are considered NEW within the time range if they started after
    // the query start time
    filter.push({
      range: {
        [_ruleDataUtils.ALERT_START]: {
          gte: start.toISOString()
        }
      }
    });
  } else if (type === AlertTypes.ONGOING) {
    // alerts are considered ONGOING within the time range if they started
    // before the query start time and they have not been recovered (no end time)
    filter = [...filter, {
      range: {
        [_ruleDataUtils.ALERT_START]: {
          lt: start.toISOString()
        }
      }
    }, {
      bool: {
        must_not: {
          exists: {
            field: _ruleDataUtils.ALERT_END
          }
        }
      }
    }];
  } else if (type === AlertTypes.RECOVERED) {
    // alerts are considered RECOVERED within the time range if they recovered
    // within the query time range
    filter.push({
      range: {
        [_ruleDataUtils.ALERT_END]: {
          gte: start.toISOString(),
          lt: end.toISOString()
        }
      }
    });
  }
  if (alertsFilter) {
    filter.push(...generateAlertsFilterDSL(alertsFilter));
  }
  return {
    size: MAX_ALERT_DOCS_TO_RETURN,
    track_total_hits: true,
    query: {
      bool: {
        filter
      }
    }
  };
};
const getQueryByScopedQueries = ({
  executionUuid,
  ruleId,
  action,
  maintenanceWindows
}) => {
  const filters = [{
    term: {
      [_ruleDataUtils.ALERT_RULE_EXECUTION_UUID]: executionUuid
    }
  }, {
    term: {
      [_ruleDataUtils.ALERT_RULE_UUID]: ruleId
    }
  }];
  if (action) {
    filters.push({
      term: {
        [_ruleDataUtils.EVENT_ACTION]: action
      }
    });
  }
  const aggs = {};
  maintenanceWindows.forEach(({
    id,
    scopedQuery
  }) => {
    var _scopedQueryFilter$bo;
    if (!scopedQuery) {
      return;
    }
    const scopedQueryFilter = generateAlertsFilterDSL({
      query: scopedQuery
    })[0];
    aggs[id] = {
      filter: {
        bool: {
          ...scopedQueryFilter.bool,
          filter: [...(((_scopedQueryFilter$bo = scopedQueryFilter.bool) === null || _scopedQueryFilter$bo === void 0 ? void 0 : _scopedQueryFilter$bo.filter) || []), ...filters]
        }
      },
      aggs: {
        alertId: {
          top_hits: {
            size: 1,
            _source: {
              includes: [_ruleDataUtils.ALERT_UUID]
            }
          }
        }
      }
    };
  });
  return {
    size: 0,
    track_total_hits: true,
    aggs: {
      ...aggs
    }
  };
};
exports.getQueryByScopedQueries = getQueryByScopedQueries;
const generateAlertsFilterDSL = alertsFilter => {
  const filter = [];
  if (alertsFilter.query) {
    filter.push(JSON.parse(alertsFilter.query.dsl));
  }
  if (alertsFilter.timeframe) {
    filter.push({
      script: {
        script: {
          source: 'params.days.contains(doc[params.datetimeField].value.withZoneSameInstant(ZoneId.of(params.timezone)).dayOfWeek.getValue())',
          params: {
            days: alertsFilter.timeframe.days.length === 0 ? _common.ISO_WEEKDAYS : alertsFilter.timeframe.days,
            timezone: alertsFilter.timeframe.timezone,
            datetimeField: _ruleDataUtils.TIMESTAMP
          }
        }
      }
    }, {
      script: {
        script: {
          source: `
              def alertsDateTime = doc[params.datetimeField].value.withZoneSameInstant(ZoneId.of(params.timezone));
              def alertsTime = LocalTime.of(alertsDateTime.getHour(), alertsDateTime.getMinute());
              def start = LocalTime.parse(params.start);
              def end = LocalTime.parse(params.end);

              if (end.isBefore(start) || end.equals(start)){ // overnight
                def dayEnd = LocalTime.parse("23:59:59");
                def dayStart = LocalTime.parse("00:00:00");
                if ((alertsTime.isAfter(start) && alertsTime.isBefore(dayEnd)) || (alertsTime.isAfter(dayStart) && alertsTime.isBefore(end))) {
                  return true;
                } else {
                  return false;
                }
              } else {
                if (alertsTime.isAfter(start) && alertsTime.isBefore(end)) {
                    return true;
                } else {
                    return false;
                }
              }
           `,
          params: {
            start: alertsFilter.timeframe.hours.start,
            end: alertsFilter.timeframe.hours.end,
            timezone: alertsFilter.timeframe.timezone,
            datetimeField: _ruleDataUtils.TIMESTAMP
          }
        }
      }
    });
  }
  return filter;
};
const getHitsWithCount = (response, formatAlert) => {
  return {
    count: response.total.value,
    data: response.hits.map(hit => {
      const {
        _id,
        _index,
        _source
      } = hit;
      const formattedSource = formatAlert && _source ? formatAlert(_source) : _source;
      const expandedSource = (0, _format_alert.expandFlattenedAlert)(formattedSource);
      return {
        _id,
        _index,
        ...expandedSource
      };
    })
  };
};
exports.getHitsWithCount = getHitsWithCount;
const getScopedQueryHitsWithIds = aggregationsResult => {
  return Object.entries(aggregationsResult || {}).reduce((result, [maintenanceWindowId, aggregation]) => {
    var _aggregation$alertId, _aggregation$alertId$;
    result[maintenanceWindowId] = (((_aggregation$alertId = aggregation.alertId) === null || _aggregation$alertId === void 0 ? void 0 : (_aggregation$alertId$ = _aggregation$alertId.hits) === null || _aggregation$alertId$ === void 0 ? void 0 : _aggregation$alertId$.hits) || []).map(hit => hit._source[_ruleDataUtils.ALERT_UUID]);
    return result;
  }, {});
};
exports.getScopedQueryHitsWithIds = getScopedQueryHitsWithIds;
const getLifecycleAlertsQueries = ({
  executionUuid,
  start,
  end,
  ruleId,
  excludedAlertInstanceIds,
  alertsFilter
}) => {
  let queryBodies;
  if (!!executionUuid) {
    queryBodies = getLifecycleAlertsQueryByExecutionUuid({
      executionUuid: executionUuid,
      ruleId,
      excludedAlertInstanceIds,
      alertsFilter
    });
  } else {
    queryBodies = getLifecycleAlertsQueryByTimeRange({
      start: start,
      end: end,
      ruleId,
      excludedAlertInstanceIds,
      alertsFilter
    });
  }
  return queryBodies;
};
exports.getLifecycleAlertsQueries = getLifecycleAlertsQueries;
const getContinualAlertsQuery = ({
  executionUuid,
  start,
  end,
  ruleId,
  excludedAlertInstanceIds,
  alertsFilter
}) => {
  let queryBody;
  if (!!executionUuid) {
    queryBody = getQueryByExecutionUuid({
      executionUuid,
      ruleId,
      excludedAlertInstanceIds,
      alertsFilter
    });
  } else {
    queryBody = getQueryByTimeRange({
      start: start,
      end: end,
      ruleId,
      excludedAlertInstanceIds,
      alertsFilter
    });
  }
  return queryBody;
};
exports.getContinualAlertsQuery = getContinualAlertsQuery;
const getMaintenanceWindowAlertsQuery = ({
  executionUuid,
  ruleId,
  action,
  maintenanceWindows
}) => {
  return getQueryByScopedQueries({
    executionUuid,
    ruleId,
    action,
    maintenanceWindows
  });
};
exports.getMaintenanceWindowAlertsQuery = getMaintenanceWindowAlertsQuery;