"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useDocumentCountStats = useDocumentCountStats;
var _react = require("react");
var _rxjs = require("rxjs");
var _i18n = require("@kbn/i18n");
var _mlStringHash = require("@kbn/ml-string-hash");
var _mlRandomSamplerUtils = require("@kbn/ml-random-sampler-utils");
var _mlErrorUtils = require("@kbn/ml-error-utils");
var _constants = require("@kbn/aiops-log-rate-analysis/constants");
var _get_document_stats = require("../get_document_stats");
var _use_aiops_app_context = require("./use_aiops_app_context");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function displayError(toastNotifications, index, err) {
  if (err.statusCode === 500) {
    var _err$error;
    toastNotifications.addError(err, {
      title: _i18n.i18n.translate('xpack.aiops.index.dataLoader.internalServerErrorMessage', {
        defaultMessage: 'Error loading data in index {index}. {message}. ' + 'The request may have timed out. Try using a smaller sample size or narrowing the time range.',
        values: {
          index,
          message: (_err$error = err.error) !== null && _err$error !== void 0 ? _err$error : err.message
        }
      })
    });
  } else {
    var _err$error2;
    toastNotifications.addError(err, {
      title: _i18n.i18n.translate('xpack.aiops.index.errorLoadingDataMessage', {
        defaultMessage: 'Error loading data in index {index}. {message}.',
        values: {
          index,
          message: (_err$error2 = err.error) !== null && _err$error2 !== void 0 ? _err$error2 : err.message
        }
      })
    });
  }
}
function useDocumentCountStats(searchParams, searchParamsCompare, lastRefresh) {
  const {
    data,
    notifications: {
      toasts
    }
  } = (0, _use_aiops_app_context.useAiopsAppContext)();
  const abortCtrl = (0, _react.useRef)(new AbortController());
  const [documentStats, setDocumentStats] = (0, _react.useState)({
    sampleProbability: 1,
    totalCount: 0
  });
  const [documentStatsCache, setDocumentStatsCache] = (0, _react.useState)({});
  const cacheKey = (0, _mlStringHash.stringHash)(`${JSON.stringify(searchParams)}_${JSON.stringify(searchParamsCompare)}`);
  const fetchDocumentCountData = (0, _react.useCallback)(async () => {
    if (!searchParams) return;
    if (documentStatsCache[cacheKey]) {
      setDocumentStats(documentStatsCache[cacheKey]);
      return;
    }
    try {
      var _totalHitsStats$total;
      abortCtrl.current = new AbortController();
      const totalHitsParams = {
        ...searchParams,
        selectedSignificantItem: undefined,
        trackTotalHits: true
      };
      const totalHitsResp = await (0, _rxjs.lastValueFrom)(data.search.search({
        params: (0, _get_document_stats.getDocumentCountStatsRequest)(totalHitsParams, undefined, true)
      }, {
        abortSignal: abortCtrl.current.signal
      }));
      const totalHitsStats = (0, _get_document_stats.processDocumentCountStats)(totalHitsResp === null || totalHitsResp === void 0 ? void 0 : totalHitsResp.rawResponse, searchParams);
      const totalCount = (_totalHitsStats$total = totalHitsStats === null || totalHitsStats === void 0 ? void 0 : totalHitsStats.totalCount) !== null && _totalHitsStats$total !== void 0 ? _totalHitsStats$total : 0;
      const randomSamplerWrapper = (0, _mlRandomSamplerUtils.createRandomSamplerWrapper)({
        totalNumDocs: totalCount,
        seed: _constants.RANDOM_SAMPLER_SEED
      });
      const resp = await (0, _rxjs.lastValueFrom)(data.search.search({
        params: (0, _get_document_stats.getDocumentCountStatsRequest)({
          ...searchParams,
          trackTotalHits: false
        }, randomSamplerWrapper, false, searchParamsCompare === undefined)
      }, {
        abortSignal: abortCtrl.current.signal
      }));
      const documentCountStats = (0, _get_document_stats.processDocumentCountStats)(resp === null || resp === void 0 ? void 0 : resp.rawResponse, searchParams, randomSamplerWrapper);
      const newStats = {
        sampleProbability: randomSamplerWrapper.probability,
        documentCountStats,
        totalCount
      };
      if (searchParamsCompare) {
        const respCompare = await (0, _rxjs.lastValueFrom)(data.search.search({
          params: (0, _get_document_stats.getDocumentCountStatsRequest)({
            ...searchParamsCompare,
            trackTotalHits: false
          }, randomSamplerWrapper)
        }, {
          abortSignal: abortCtrl.current.signal
        }));
        const documentCountStatsCompare = (0, _get_document_stats.processDocumentCountStats)(respCompare === null || respCompare === void 0 ? void 0 : respCompare.rawResponse, searchParamsCompare, randomSamplerWrapper);
        newStats.documentCountStatsCompare = documentCountStatsCompare;
        newStats.totalCount = totalCount;
      }
      setDocumentStats(newStats);
      setDocumentStatsCache({
        ...documentStatsCache,
        [cacheKey]: newStats
      });
    } catch (error) {
      // An `AbortError` gets triggered when a user cancels a request by navigating away, we need to ignore these errors.
      if (error.name !== 'AbortError') {
        displayError(toasts, searchParams.index, (0, _mlErrorUtils.extractErrorProperties)(error));
      }
    }
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [data === null || data === void 0 ? void 0 : data.search, documentStatsCache, cacheKey]);
  (0, _react.useEffect)(function getDocumentCountData() {
    fetchDocumentCountData();
    return () => abortCtrl.current.abort();
  }, [fetchDocumentCountData, lastRefresh]);

  // Clear the document count stats cache when the outer page (date picker/search bar) triggers a refresh.
  (0, _react.useEffect)(() => {
    setDocumentStatsCache({});
  }, [lastRefresh]);
  return documentStats;
}