"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTokenCountFromInvokeAsyncIterator = getTokenCountFromInvokeAsyncIterator;
var _gptTokenizer = require("gpt-tokenizer");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Takes the OpenAI and Bedrock `invokeStream` sub action response stream and the request messages array as inputs.
 * Uses gpt-tokenizer encoding to calculate the number of tokens in the prompt and completion parts of the response stream
 * Returns an object containing the total, prompt, and completion token counts.
 * @param streamIterable the response iterator from the `invokeAsyncIterator` sub action
 * @param body the request messages array
 * @param logger the logger
 */
async function getTokenCountFromInvokeAsyncIterator({
  streamIterable,
  body,
  logger
}) {
  const chatCompletionRequest = body;

  // per https://github.com/openai/openai-cookbook/blob/main/examples/How_to_count_tokens_with_tiktoken.ipynb
  const promptTokens = (0, _gptTokenizer.encode)(chatCompletionRequest.messages.map(msg => `<|start|>${msg.role}\n${msg.content}\n${'name' in msg ? msg.name : 'function_call' in msg && msg.function_call ? msg.function_call.name + '\n' + msg.function_call.arguments : ''}<|end|>`).join('\n')).length;
  const parsedResponse = await parseOpenAIStream(streamIterable, logger);
  const completionTokens = (0, _gptTokenizer.encode)(parsedResponse).length;
  return {
    prompt: promptTokens,
    completion: completionTokens,
    total: promptTokens + completionTokens
  };
}
const parseOpenAIStream = async (streamIterable, logger) => {
  let responseBody = '';
  try {
    for await (const data of streamIterable) {
      var _data$choices, _delta$content;
      if (!data) continue;
      const choice = data === null || data === void 0 ? void 0 : (_data$choices = data.choices) === null || _data$choices === void 0 ? void 0 : _data$choices[0];
      if (!choice) continue;
      const {
        delta
      } = choice;
      if (!delta) continue;
      const chunk = (_delta$content = delta.content) !== null && _delta$content !== void 0 ? _delta$content : '';
      if (typeof chunk !== 'string') {
        logger.warn('Received non-string content from OpenAI. This is currently not supported.');
        continue;
      }
      responseBody += chunk;
    }
  } catch (e) {
    logger.error('An error occurred while calculating streaming response tokens');
    logger.error(e);
  }
  return responseBody;
};