"use strict";
Object.defineProperty(exports, "__esModule", { value: true });
exports.renderBullet = void 0;
const sub_types_1 = require("./sub_types");
const colors_1 = require("../../../../common/colors");
const text_utils_1 = require("../../../../common/text_utils");
const canvas_1 = require("../../../../renderers/canvas");
const debug_1 = require("../../../xy_chart/renderer/canvas/utils/debug");
const spec_1 = require("../../spec");
const theme_1 = require("../../theme");
function renderBullet(ctx, dpr, props) {
    const { debug, style, dimensions, activeValues, spec, backgroundColor } = props;
    (0, canvas_1.withContext)(ctx, (ctx) => {
        ctx.scale(dpr, dpr);
        (0, canvas_1.clearCanvas)(ctx, backgroundColor);
        if (!spec || dimensions.shouldRenderMetric) {
            return;
        }
        ctx.fillStyle = backgroundColor;
        dimensions.rows.forEach((row, rowIndex) => row.forEach((bulletGraph, columnIndex) => {
            if (!bulletGraph)
                return;
            const { panel, multiline } = bulletGraph;
            (0, canvas_1.withContext)(ctx, (ctx) => {
                var _a;
                const verticalAlignment = dimensions.layoutAlignment[rowIndex];
                const activeValue = (_a = activeValues === null || activeValues === void 0 ? void 0 : activeValues[rowIndex]) === null || _a === void 0 ? void 0 : _a[columnIndex];
                if (debug) {
                    (0, debug_1.renderDebugRect)(ctx, panel);
                }
                ctx.translate(panel.x, panel.y);
                ctx.strokeStyle = style.border;
                if (row.length > 1 && columnIndex < row.length - 1) {
                    ctx.beginPath();
                    ctx.moveTo(panel.width, 0);
                    ctx.lineTo(panel.width, panel.height);
                    ctx.stroke();
                }
                if (dimensions.rows.length > 1 && columnIndex < dimensions.rows.length) {
                    ctx.beginPath();
                    ctx.moveTo(0, panel.height);
                    ctx.lineTo(panel.width, panel.height);
                    ctx.stroke();
                }
                ctx.translate(theme_1.HEADER_PADDING.left, theme_1.HEADER_PADDING.top);
                ctx.textBaseline = 'alphabetic';
                const MAX_TARGET_VALUE_ASCENT = (0, theme_1.getMaxTargetValueAssent)(bulletGraph.target);
                const commonYBaseline = Math.max((0, theme_1.getTextAscentHeight)(theme_1.TITLE_FONT_SIZE, verticalAlignment.maxTitleRows, theme_1.TITLE_LINE_SPACING) +
                    (verticalAlignment.maxSubtitleRows > 0 ? theme_1.FONT_PADDING : 0) +
                    (0, theme_1.getTextAscentHeight)(theme_1.SUBTITLE_FONT_SIZE, verticalAlignment.maxSubtitleRows), verticalAlignment.multiline ? 0 : MAX_TARGET_VALUE_ASCENT);
                ctx.fillStyle = props.style.textColor;
                ctx.textAlign = 'start';
                ctx.font = (0, text_utils_1.cssFontShorthand)(theme_1.TITLE_FONT, theme_1.TITLE_FONT_SIZE);
                const titleYBaseline = commonYBaseline -
                    (0, theme_1.getTextAscentHeight)(theme_1.SUBTITLE_FONT_SIZE, verticalAlignment.maxSubtitleRows) -
                    (verticalAlignment.maxSubtitleRows > 0 ? theme_1.FONT_PADDING : 0);
                bulletGraph.title
                    .slice()
                    .reverse()
                    .forEach((titleLine, lineIndex) => {
                    const y = titleYBaseline - lineIndex * ((0, theme_1.getTextAscentHeight)(theme_1.TITLE_FONT_SIZE) + theme_1.TITLE_LINE_SPACING);
                    ctx.fillText(titleLine, 0, y);
                });
                if (bulletGraph.subtitle) {
                    ctx.font = (0, text_utils_1.cssFontShorthand)(theme_1.SUBTITLE_FONT, theme_1.SUBTITLE_FONT_SIZE);
                    ctx.fillText(bulletGraph.subtitle, 0, commonYBaseline);
                }
                ctx.font = (0, text_utils_1.cssFontShorthand)(theme_1.VALUE_FONT, theme_1.VALUE_FONT_SIZE);
                if (!multiline)
                    ctx.textAlign = 'end';
                {
                    const y = commonYBaseline + (multiline ? MAX_TARGET_VALUE_ASCENT + theme_1.FONT_PADDING : 0);
                    const x = multiline ? 0 : bulletGraph.header.width - bulletGraph.targetWidth;
                    ctx.fillText(bulletGraph.value, x, y);
                }
                if (bulletGraph.target) {
                    ctx.font = (0, text_utils_1.cssFontShorthand)(theme_1.TARGET_FONT, theme_1.TARGET_FONT_SIZE);
                    if (!multiline)
                        ctx.textAlign = 'end';
                    const x = multiline ? bulletGraph.valueWidth : bulletGraph.header.width;
                    const y = commonYBaseline + (multiline ? MAX_TARGET_VALUE_ASCENT + theme_1.FONT_PADDING : 0);
                    ctx.fillText(bulletGraph.target, x, y);
                }
                ctx.translate(-theme_1.HEADER_PADDING.left, -theme_1.HEADER_PADDING.top);
                const { graphArea } = bulletGraph;
                if (spec.subtype !== spec_1.BulletSubtype.horizontal) {
                    ctx.strokeStyle = style.border;
                    ctx.beginPath();
                    ctx.moveTo(theme_1.HEADER_PADDING.left, graphArea.origin.y);
                    ctx.lineTo(panel.width - theme_1.HEADER_PADDING.right, graphArea.origin.y);
                    ctx.stroke();
                }
                (0, canvas_1.withContext)(ctx, (ctx) => {
                    ctx.translate(graphArea.origin.x, graphArea.origin.y);
                    if (spec.subtype === spec_1.BulletSubtype.horizontal) {
                        (0, sub_types_1.horizontalBullet)(ctx, bulletGraph, style, backgroundColor, activeValue);
                    }
                    else if (spec.subtype === spec_1.BulletSubtype.vertical) {
                        (0, sub_types_1.verticalBullet)(ctx, bulletGraph, style, backgroundColor, activeValue);
                    }
                    else {
                        (0, sub_types_1.angularBullet)(ctx, bulletGraph, style, backgroundColor, spec, debug, activeValue);
                    }
                });
                if (debug) {
                    (0, canvas_1.withContext)(ctx, (ctx) => {
                        ctx.translate(graphArea.origin.x, graphArea.origin.y);
                        (0, debug_1.renderDebugRect)(ctx, {
                            ...graphArea.size,
                            x: 0,
                            y: 0,
                        }, 0, { color: colors_1.Colors.Transparent.rgba });
                        (0, debug_1.renderDebugPoint)(ctx, 0, 0);
                        (0, debug_1.renderDebugPoint)(ctx, graphArea.size.width / 2, graphArea.size.height / 2);
                    });
                }
            });
        }));
    });
}
exports.renderBullet = renderBullet;
//# sourceMappingURL=bullet.js.map