import { Serializable } from "./load/serializable.js";
import { HumanMessage, AIMessage } from "./messages/index.js";
/**
 * Base class for all chat message histories. All chat message histories
 * should extend this class.
 */
export class BaseChatMessageHistory extends Serializable {
}
/**
 * Base class for all list chat message histories. All list chat message
 * histories should extend this class.
 */
export class BaseListChatMessageHistory extends Serializable {
    /**
     * This is a convenience method for adding a human message string to the store.
     * Please note that this is a convenience method. Code should favor the
     * bulk addMessages interface instead to save on round-trips to the underlying
     * persistence layer.
     * This method may be deprecated in a future release.
     */
    addUserMessage(message) {
        return this.addMessage(new HumanMessage(message));
    }
    /** @deprecated Use addAIMessage instead */
    addAIChatMessage(message) {
        return this.addMessage(new AIMessage(message));
    }
    /**
     * This is a convenience method for adding an AI message string to the store.
     * Please note that this is a convenience method. Code should favor the bulk
     * addMessages interface instead to save on round-trips to the underlying
     * persistence layer.
     * This method may be deprecated in a future release.
     */
    addAIMessage(message) {
        return this.addMessage(new AIMessage(message));
    }
    /**
     * Add a list of messages.
     *
     * Implementations should override this method to handle bulk addition of messages
     * in an efficient manner to avoid unnecessary round-trips to the underlying store.
     *
     * @param messages - A list of BaseMessage objects to store.
     */
    async addMessages(messages) {
        for (const message of messages) {
            await this.addMessage(message);
        }
    }
    /**
     * Remove all messages from the store.
     */
    clear() {
        throw new Error("Not implemented.");
    }
}
