"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getTypeFromAction = getTypeFromAction;
exports.onWatchSave = onWatchSave;
exports.saveWatch = saveWatch;
var _i18n = require("@kbn/i18n");
var _lodash = require("lodash");
var _constants = require("../../../../common/constants");
var _api = require("../../lib/api");
var _navigation = require("../../lib/navigation");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * Get the type from an action where a key defines its type.
 * eg: { email: { ... } } | { slack: { ... } }
 */
function getTypeFromAction(action) {
  const actionKeys = Object.keys(action);
  let type;
  Object.keys(_constants.ACTION_TYPES).forEach(k => {
    if (actionKeys.includes(_constants.ACTION_TYPES[k])) {
      type = _constants.ACTION_TYPES[k];
    }
  });
  return type ? type : _constants.ACTION_TYPES.UNKNOWN;
}
function getPropsFromAction(type, action) {
  if (type === _constants.ACTION_TYPES.SLACK) {
    // Slack action has its props inside the "message" object
    return action[type].message;
  }
  if (type === _constants.ACTION_TYPES.JIRA) {
    // Jira action has its required props inside the "fields" object
    const jiraAction = {};
    jiraAction.projectKey = (0, _lodash.get)(action[type], 'fields.project.key');
    jiraAction.issueType = (0, _lodash.get)(action[type], 'fields.issuetype.name');
    jiraAction.summary = (0, _lodash.get)(action[type], 'fields.summary');
    return jiraAction;
  }
  return action[type];
}

/**
 * Actions instances are not automatically added to the Watch _actions_ Array
 * when we add them in the JSON watch editor. This method takes takes care of it.
 */
function createActionsForWatch(watchInstance) {
  watchInstance.resetActions();
  Object.keys(watchInstance.watch.actions).forEach(k => {
    const action = watchInstance.watch.actions[k];
    const type = getTypeFromAction(action);
    const actionProps = {
      ...getPropsFromAction(type, action),
      ignoreDefaults: true
    };
    watchInstance.createAction(type, actionProps);
  });
  return watchInstance;
}
async function saveWatch(watch, toasts) {
  try {
    await (0, _api.createWatch)(watch);
    toasts.addSuccess(watch.isNew ? _i18n.i18n.translate('xpack.watcher.sections.watchEdit.json.createSuccessNotificationText', {
      defaultMessage: "Created '{watchDisplayName}'",
      values: {
        watchDisplayName: watch.displayName
      }
    }) : _i18n.i18n.translate('xpack.watcher.sections.watchEdit.json.saveSuccessNotificationText', {
      defaultMessage: "Saved '{watchDisplayName}'",
      values: {
        watchDisplayName: watch.displayName
      }
    }));
    (0, _navigation.goToWatchList)();
  } catch (error) {
    var _error$response$data;
    return {
      error: (_error$response$data = error === null || error === void 0 ? void 0 : error.response.data) !== null && _error$response$data !== void 0 ? _error$response$data : error.body || error
    };
  }
}
async function onWatchSave(watch, toasts) {
  const watchActions = watch.watch && watch.watch.actions;
  const watchData = watchActions ? createActionsForWatch(watch) : watch;
  if (watchData.type === _constants.WATCH_TYPES.JSON) {
    const actionsErrors = watchData.actions.reduce((actionsErrorsAcc, action) => {
      if (action.validate) {
        const errors = action.validate();
        const errorKeys = Object.keys(errors);
        const hasErrors = !!errorKeys.find(errorKey => errors[errorKey].length >= 1);
        if (!hasErrors) {
          return actionsErrorsAcc;
        }
        const newErrors = errorKeys.map(errorKey => errors[errorKey]);
        const newErrorsFlattened = newErrors && newErrors.length ? [].concat(...newErrors) : [];
        return [...actionsErrorsAcc, ...newErrorsFlattened];
      }
      return actionsErrorsAcc;
    }, []);
    if (actionsErrors.length > 0) {
      return {
        error: {
          data: {
            message: actionsErrors,
            error: 'validation'
          }
        }
      };
    }
    return saveWatch(watchData, toasts);
  }
  return saveWatch(watchData, toasts);
}