"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.Vis = void 0;
var _lodash = _interopRequireDefault(require("lodash"));
var _d = _interopRequireDefault(require("d3"));
var _events = require("events");
var _errors = require("./errors");
var _vis_config = require("./lib/vis_config");
var _handler = require("./lib/handler");
var _common = require("../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

/**
 * Creates the visualizations.
 *
 * @class Vis
 * @constructor
 * @param element {HTMLElement} jQuery selected HTML element
 * @param config {Object} Parameters that define the chart type and chart options
 */
class Vis extends _events.EventEmitter {
  constructor(element, visConfigArgs, core, charts) {
    super();
    this.element = element.get ? element.get(0) : element;
    this.visConfigArgs = _lodash.default.cloneDeep(visConfigArgs);
    this.visConfigArgs.heatmapMaxBuckets = core.uiSettings.get(_common.HEATMAP_MAX_BUCKETS_SETTING);
    this.charts = charts;
    this.uiSettings = core.uiSettings;
  }
  hasLegend() {
    return this.visConfigArgs.addLegend;
  }
  initVisConfig(data, uiState) {
    this.data = data;
    this.uiState = uiState;
    this.visConfig = new _vis_config.VisConfig(this.visConfigArgs, this.data, this.uiState, this.element, this.charts.legacyColors.createColorLookupFunction.bind(this.charts.legacyColors));
  }

  /**
   * Renders the visualization
   *
   * @method render
   * @param data {Object} Elasticsearch query results
   */
  render(data, uiState) {
    if (!data) {
      throw new Error('No valid data!');
    }
    if (this.handler) {
      this.data = null;
      this._runOnHandler('destroy');
    }
    this.initVisConfig(data, uiState);
    this.handler = new _handler.Handler(this, this.visConfig, this.uiSettings);
    this._runOnHandler('render');
  }
  getLegendLabels() {
    return this.visConfig ? this.visConfig.get('legend.labels', null) : null;
  }
  getLegendColors() {
    return this.visConfig ? this.visConfig.get('legend.colors', null) : null;
  }
  _runOnHandler(method) {
    try {
      this.handler[method]();
    } catch (error) {
      if (error instanceof _errors.VislibError) {
        error.displayToScreen(this.handler);
      } else {
        throw error;
      }
    }
  }

  /**
   * Destroys the visualization
   * Removes chart and all elements associated with it.
   * Removes chart and all elements associated with it.
   * Remove event listeners and pass destroy call down to owned objects.
   *
   * @method destroy
   */
  destroy() {
    const selection = _d.default.select(this.element).select('.visWrapper');
    if (this.handler) this._runOnHandler('destroy');
    selection.remove();
  }

  /**
   * Sets attributes on the visualization
   *
   * @method set
   * @param name {String} An attribute name
   * @param val {*} Value to which the attribute name is set
   */
  set(name, val) {
    this.visConfigArgs[name] = val;
    this.render(this.data, this.uiState);
  }

  /**
   * Gets attributes from the visualization
   *
   * @method get
   * @param name {String} An attribute name
   * @returns {*} The value of the attribute name
   */
  get(name) {
    return this.visConfig.get(name);
  }

  /**
   * Turns on event listeners.
   *
   * @param event {String}
   * @param listener{Function}
   * @returns {*}
   */
  on(event, listener) {
    const first = this.listenerCount(event) === 0;
    const ret = _events.EventEmitter.prototype.on.call(this, event, listener);
    const added = this.listenerCount(event) > 0;

    // if this is the first listener added for the event
    // enable the event in the handler
    if (first && added && this.handler) this.handler.enable(event);
    return ret;
  }

  /**
   * Turns off event listeners.
   *
   * @param event {String}
   * @param listener{Function}
   * @returns {*}
   */
  off(event, listener) {
    const last = this.listenerCount(event) === 1;
    const ret = _events.EventEmitter.prototype.off.call(this, event, listener);
    const removed = this.listenerCount(event) === 0;

    // Once all listeners are removed, disable the events in the handler
    if (last && removed && this.handler) this.handler.disable(event);
    return ret;
  }
  removeAllListeners(event) {
    const ret = _events.EventEmitter.prototype.removeAllListeners.call(this, event);
    this.handler.disable(event);
    return ret;
  }
}
exports.Vis = Vis;