"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.UrlForwardingPlugin = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _forward_app = require("./forward_app");
var _navigate_to_legacy_kibana_url = require("./forward_app/navigate_to_legacy_kibana_url");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

class UrlForwardingPlugin {
  constructor() {
    (0, _defineProperty2.default)(this, "forwardDefinitions", []);
  }
  setup(core) {
    core.application.register((0, _forward_app.createLegacyUrlForwardApp)(core, this.forwardDefinitions));
    return {
      /**
       * Forwards URLs within the legacy `kibana` app to a new platform application.
       *
       * @param legacyAppId The name of the old app to forward URLs from
       * @param newAppId The name of the new app that handles the URLs now
       * @param rewritePath Function to rewrite the legacy sub path of the app to the new path in the core app.
       *        If none is provided, it will just strip the prefix of the legacyAppId away
       *
       * path into the new path
       *
       * Example usage:
       * ```
       * urlForwarding.forwardApp(
       *   'old',
       *   'new',
       *   path => {
       *     const [, id] = /old/item\/(.*)$/.exec(path) || [];
       *     if (!id) {
       *       return '#/home';
       *     }
       *     return '#/items/${id}';
       *  }
       * );
       * ```
       * This will cause the following redirects:
       *
       * * app/kibana#/old/ -> app/new#/home
       * * app/kibana#/old/item/123 -> app/new#/items/123
       *
       */
      forwardApp: (legacyAppId, newAppId, rewritePath) => {
        this.forwardDefinitions.push({
          legacyAppId,
          newAppId,
          rewritePath: rewritePath || (path => `#${path.replace(`/${legacyAppId}`, '') || '/'}`)
        });
      }
    };
  }
  start({
    application,
    http: {
      basePath
    }
  }) {
    return {
      /**
       * Resolves the provided hash using the registered forwards and navigates to the target app.
       * If a navigation happened, `{ navigated: true }` will be returned.
       * If no matching forward is found, `{ navigated: false }` will be returned.
       * @param hash
       */
      navigateToLegacyKibanaUrl: hash => {
        return (0, _navigate_to_legacy_kibana_url.navigateToLegacyKibanaUrl)(hash, this.forwardDefinitions, basePath, application);
      },
      /**
       * @deprecated
       * Just exported for wiring up with legacy platform, should not be used.
       */
      getForwards: () => this.forwardDefinitions
    };
  }
}
exports.UrlForwardingPlugin = UrlForwardingPlugin;