"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.WEBHOOK_ACTION_ID = exports.TEAMS_ACTION_ID = exports.SLACK_WEBHOOK_ACTION_ID = exports.SERVICE_NOW_ACTION_ID = exports.SERVER_LOG_ACTION_ID = exports.PAGER_DUTY_ACTION_ID = exports.JIRA_ACTION_ID = exports.INDEX_ACTION_ID = exports.EMAIL_ACTION_ID = void 0;
exports.populateAlertActions = populateAlertActions;
var _uuid = require("uuid");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const SLACK_WEBHOOK_ACTION_ID = exports.SLACK_WEBHOOK_ACTION_ID = '.slack';
const PAGER_DUTY_ACTION_ID = exports.PAGER_DUTY_ACTION_ID = '.pagerduty';
const SERVER_LOG_ACTION_ID = exports.SERVER_LOG_ACTION_ID = '.server-log';
const INDEX_ACTION_ID = exports.INDEX_ACTION_ID = '.index';
const TEAMS_ACTION_ID = exports.TEAMS_ACTION_ID = '.teams';
const SERVICE_NOW_ACTION_ID = exports.SERVICE_NOW_ACTION_ID = '.servicenow';
const JIRA_ACTION_ID = exports.JIRA_ACTION_ID = '.jira';
const WEBHOOK_ACTION_ID = exports.WEBHOOK_ACTION_ID = '.webhook';
const EMAIL_ACTION_ID = exports.EMAIL_ACTION_ID = '.email';
function populateAlertActions({
  defaultActions,
  defaultEmail,
  groupId,
  translations
}) {
  const actions = [];
  defaultActions.forEach(aId => {
    const action = {
      id: aId.id,
      group: groupId,
      params: {},
      frequency: undefined
    };
    const recoveredAction = {
      id: aId.id,
      group: 'recovered',
      params: {
        message: translations.defaultRecoveryMessage
      },
      frequency: undefined
    };
    switch (aId.actionTypeId) {
      case PAGER_DUTY_ACTION_ID:
        const dedupKey = (0, _uuid.v4)();
        action.params = getPagerDutyActionParams(translations, dedupKey);
        recoveredAction.params = getPagerDutyActionParams(translations, dedupKey, true);
        actions.push(recoveredAction);
        break;
      case SERVER_LOG_ACTION_ID:
        action.params = getServerLogActionParams(translations);
        recoveredAction.params = getServerLogActionParams(translations, true);
        actions.push(recoveredAction);
        break;
      case INDEX_ACTION_ID:
        action.params = getIndexActionParams(translations, false);
        recoveredAction.params = getIndexActionParams(translations, true);
        actions.push(recoveredAction);
        break;
      case SERVICE_NOW_ACTION_ID:
        action.params = getServiceNowActionParams(translations);
        // Recovery action for service now is not implemented yet
        break;
      case JIRA_ACTION_ID:
        action.params = getJiraActionParams(translations);
        // Recovery action for Jira is not implemented yet
        break;
      case WEBHOOK_ACTION_ID:
        action.params = getWebhookActionParams(translations);
        recoveredAction.params = getWebhookActionParams(translations, true);
        actions.push(recoveredAction);
        break;
      case SLACK_WEBHOOK_ACTION_ID:
      case TEAMS_ACTION_ID:
        action.params = {
          message: translations.defaultActionMessage
        };
        actions.push(recoveredAction);
        break;
      case EMAIL_ACTION_ID:
        if (defaultEmail) {
          action.params = getEmailActionParams(translations, defaultEmail);
          recoveredAction.params = getEmailActionParams(translations, defaultEmail, true);
          actions.push(recoveredAction);
        }
        break;
      default:
        action.params = {
          message: translations.defaultActionMessage
        };
    }
    actions.push(action);
  });
  return actions;
}
function getIndexActionParams(translations, recovery = false) {
  if (recovery) {
    return {
      documents: [{
        monitorName: '{{context.monitorName}}',
        monitorUrl: '{{{context.monitorUrl}}}',
        statusMessage: translations.defaultRecoveryMessage,
        latestErrorMessage: '',
        observerLocation: '{{context.observerLocation}}'
      }],
      indexOverride: null
    };
  }
  return {
    documents: [{
      monitorName: '{{context.monitorName}}',
      monitorUrl: '{{{context.monitorUrl}}}',
      statusMessage: '{{{context.statusMessage}}}',
      latestErrorMessage: '{{{context.latestErrorMessage}}}',
      observerLocation: '{{context.observerLocation}}'
    }],
    indexOverride: null
  };
}
function getServerLogActionParams({
  defaultActionMessage,
  defaultRecoveryMessage
}, recovery = false) {
  if (recovery) {
    return {
      level: 'info',
      message: defaultRecoveryMessage
    };
  }
  return {
    level: 'warn',
    message: defaultActionMessage
  };
}
function getWebhookActionParams({
  defaultActionMessage,
  defaultRecoveryMessage
}, recovery = false) {
  return {
    body: recovery ? defaultRecoveryMessage : defaultActionMessage
  };
}
function getPagerDutyActionParams({
  defaultActionMessage,
  defaultRecoveryMessage
}, dedupKey, recovery = false) {
  if (recovery) {
    return {
      dedupKey,
      eventAction: 'resolve',
      summary: defaultRecoveryMessage
    };
  }
  return {
    dedupKey,
    eventAction: 'trigger',
    severity: 'error',
    summary: defaultActionMessage
  };
}
function getServiceNowActionParams({
  defaultActionMessage
}) {
  return {
    subAction: 'pushToService',
    subActionParams: {
      incident: {
        short_description: defaultActionMessage,
        description: defaultActionMessage,
        impact: '2',
        severity: '2',
        urgency: '2',
        category: null,
        subcategory: null,
        externalId: null,
        correlation_id: null,
        correlation_display: null
      },
      comments: []
    }
  };
}
function getJiraActionParams({
  defaultActionMessage
}) {
  return {
    subAction: 'pushToService',
    subActionParams: {
      incident: {
        summary: defaultActionMessage,
        externalId: null,
        description: defaultActionMessage,
        issueType: null,
        priority: '2',
        labels: null,
        parent: null,
        otherFields: null
      },
      comments: []
    }
  };
}
function getEmailActionParams({
  defaultActionMessage,
  defaultSubjectMessage,
  defaultRecoverySubjectMessage,
  defaultRecoveryMessage
}, defaultEmail, isRecovery) {
  var _defaultEmail$cc, _defaultEmail$bcc;
  return {
    to: defaultEmail.to,
    subject: isRecovery ? defaultRecoverySubjectMessage : defaultSubjectMessage,
    message: isRecovery ? defaultRecoveryMessage : defaultActionMessage,
    messageHTML: null,
    cc: (_defaultEmail$cc = defaultEmail.cc) !== null && _defaultEmail$cc !== void 0 ? _defaultEmail$cc : [],
    bcc: (_defaultEmail$bcc = defaultEmail.bcc) !== null && _defaultEmail$bcc !== void 0 ? _defaultEmail$bcc : [],
    kibanaFooterLink: {
      path: '',
      text: ''
    }
  };
}