"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getRuleReducer = void 0;
var _lodash = require("lodash");
var _fp = require("lodash/fp");
var _constants = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getRuleReducer = actionTypeRegistry => (state, action) => {
  const {
    rule
  } = state;
  switch (action.command.type) {
    case 'setRule':
      {
        const {
          key,
          value
        } = action.payload;
        if (key === 'rule') {
          return {
            ...state,
            rule: value
          };
        } else {
          return state;
        }
      }
    case 'setProperty':
      {
        const {
          key,
          value
        } = action.payload;
        if ((0, _lodash.isEqual)(rule[key], value)) {
          return state;
        } else {
          return {
            ...state,
            rule: {
              ...rule,
              [key]: value
            }
          };
        }
      }
    case 'setScheduleProperty':
      {
        const {
          key,
          value
        } = action.payload;
        if (rule.schedule && (0, _lodash.isEqual)(rule.schedule[key], value)) {
          return state;
        } else {
          return {
            ...state,
            rule: {
              ...rule,
              schedule: {
                ...rule.schedule,
                [key]: value
              }
            }
          };
        }
      }
    case 'setRuleParams':
      {
        const {
          key,
          value
        } = action.payload;
        if ((0, _lodash.isEqual)(rule.params[key], value)) {
          return state;
        } else {
          return {
            ...state,
            rule: {
              ...rule,
              params: {
                ...rule.params,
                [key]: value
              }
            }
          };
        }
      }
    case 'setRuleActionParams':
      {
        const {
          key,
          value,
          index
        } = action.payload;
        if (index === undefined || rule.actions[index] == null || !!rule.actions[index][key] && (0, _lodash.isEqual)(rule.actions[index][key], value)) {
          return state;
        } else {
          const oldAction = rule.actions.splice(index, 1)[0];
          const updatedAction = {
            ...oldAction,
            params: {
              ...oldAction.params,
              [key]: value
            }
          };
          rule.actions.splice(index, 0, updatedAction);
          return {
            ...state,
            rule: {
              ...rule,
              actions: [...rule.actions]
            }
          };
        }
      }
    case 'setRuleActionFrequency':
      {
        const {
          key,
          value,
          index
        } = action.payload;
        if (index === undefined || rule.actions[index] == null || !!rule.actions[index][key] && (0, _lodash.isEqual)(rule.actions[index][key], value)) {
          return state;
        } else {
          var _oldSanitizedAction$f;
          const oldAction = rule.actions.splice(index, 1)[0];
          if (actionTypeRegistry.get(oldAction.actionTypeId).isSystemActionType) {
            return state;
          }
          const oldSanitizedAction = oldAction;
          const updatedAction = {
            ...oldSanitizedAction,
            frequency: {
              ...((_oldSanitizedAction$f = oldSanitizedAction === null || oldSanitizedAction === void 0 ? void 0 : oldSanitizedAction.frequency) !== null && _oldSanitizedAction$f !== void 0 ? _oldSanitizedAction$f : _constants.DEFAULT_FREQUENCY),
              [key]: value
            }
          };
          rule.actions.splice(index, 0, updatedAction);
          return {
            ...state,
            rule: {
              ...rule,
              actions: [...rule.actions]
            }
          };
        }
      }
    case 'setRuleActionAlertsFilter':
      {
        const {
          key,
          value,
          index
        } = action.payload;
        if (index === undefined || rule.actions[index] == null) {
          return state;
        } else {
          const oldAction = rule.actions.splice(index, 1)[0];
          if (actionTypeRegistry.get(oldAction.actionTypeId).isSystemActionType) {
            return state;
          }
          const oldSanitizedAction = oldAction;
          if (oldSanitizedAction.alertsFilter && (0, _lodash.isEqual)(oldSanitizedAction.alertsFilter[key], value)) return state;
          const {
            alertsFilter,
            ...rest
          } = oldSanitizedAction;
          const updatedAlertsFilter = {
            ...alertsFilter,
            [key]: value
          };
          const updatedAction = {
            ...rest,
            ...(!(0, _fp.isEmpty)(updatedAlertsFilter) ? {
              alertsFilter: updatedAlertsFilter
            } : {})
          };
          rule.actions.splice(index, 0, updatedAction);
          return {
            ...state,
            rule: {
              ...rule,
              actions: [...rule.actions]
            }
          };
        }
      }
    case 'setRuleActionProperty':
      {
        const {
          key,
          value,
          index
        } = action.payload;
        if (index === undefined || (0, _lodash.isEqual)(rule.actions[index][key], value)) {
          return state;
        } else {
          const oldAction = rule.actions.splice(index, 1)[0];
          const updatedAction = {
            ...oldAction,
            [key]: value
          };
          rule.actions.splice(index, 0, updatedAction);
          return {
            ...state,
            rule: {
              ...rule,
              actions: [...rule.actions]
            }
          };
        }
      }
    case 'setAlertDelayProperty':
      {
        const {
          key,
          value
        } = action.payload;
        if (rule.alertDelay && (0, _lodash.isEqual)(rule.alertDelay[key], value)) {
          return state;
        } else {
          return {
            ...state,
            rule: {
              ...rule,
              alertDelay: {
                ...rule.alertDelay,
                [key]: value
              }
            }
          };
        }
      }
  }
};
exports.getRuleReducer = getRuleReducer;