"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useIndexData = void 0;
var _react = require("react");
var _ebtTools = require("@kbn/ebt-tools");
var _mlRuntimeFieldUtils = require("@kbn/ml-runtime-field-utils");
var _mlQueryUtils = require("@kbn/ml-query-utils");
var _mlDataGrid = require("@kbn/ml-data-grid");
var _field_utils = require("../../../common/utils/field_utils");
var _errors = require("../../../common/utils/errors");
var _app_dependencies = require("../app_dependencies");
var _use_get_histograms_for_fields = require("./use_get_histograms_for_fields");
var _use_data_search = require("./use_data_search");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const useIndexData = (dataView, query, combinedRuntimeMappings, timeRangeMs, populatedFields) => {
  const {
    analytics
  } = (0, _app_dependencies.useAppDependencies)();

  // Store the performance metric's start time using a ref
  // to be able to track it across rerenders.
  const loadIndexDataStartTime = (0, _react.useRef)(window.performance.now());
  const indexPattern = (0, _react.useMemo)(() => dataView.getIndexPattern(), [dataView]);
  const toastNotifications = (0, _app_dependencies.useToastNotifications)();
  const baseFilterCriteria = (0, _mlQueryUtils.buildBaseFilterCriteria)(dataView.timeFieldName, timeRangeMs === null || timeRangeMs === void 0 ? void 0 : timeRangeMs.from, timeRangeMs === null || timeRangeMs === void 0 ? void 0 : timeRangeMs.to, query);
  const defaultQuery = (0, _react.useMemo)(() => timeRangeMs && dataView.timeFieldName ? baseFilterCriteria[0] : _mlQueryUtils.matchAllQuery, [baseFilterCriteria, dataView, timeRangeMs]);
  const queryWithBaseFilterCriteria = {
    bool: {
      filter: baseFilterCriteria
    }
  };

  // Fetch 500 random documents to determine populated fields.
  // This is a workaround to avoid passing potentially thousands of unpopulated fields
  // (for example, as part of filebeat/metricbeat/ECS based indices)
  // to the data grid component which would significantly slow down the page.
  const {
    error: dataViewFieldsError,
    data: dataViewFieldsData,
    isError: dataViewFieldsIsError,
    isLoading: dataViewFieldsIsLoading
  } = (0, _use_data_search.useDataSearch)({
    index: indexPattern,
    body: {
      fields: ['*'],
      _source: false,
      query: {
        function_score: {
          query: defaultQuery,
          random_score: {}
        }
      },
      size: 500
    }
  },
  // Check whether fetching should be enabled
  // If populatedFields are not provided, make own request to calculate
  !Array.isArray(populatedFields) && !(dataView.timeFieldName !== undefined && timeRangeMs === undefined));
  (0, _react.useEffect)(() => {
    if (dataViewFieldsIsLoading && !dataViewFieldsIsError) {
      setErrorMessage('');
      setStatus(_mlDataGrid.INDEX_STATUS.LOADING);
    } else if (dataViewFieldsError !== null) {
      setErrorMessage((0, _errors.getErrorMessage)(dataViewFieldsError));
      setStatus(_mlDataGrid.INDEX_STATUS.ERROR);
    } else if (!dataViewFieldsIsLoading && !dataViewFieldsIsError && dataViewFieldsData !== undefined) {
      const isCrossClusterSearch = indexPattern.includes(':');
      const isMissingFields = dataViewFieldsData.hits.hits.every(d => typeof d.fields === 'undefined');
      setCcsWarning(isCrossClusterSearch && isMissingFields);
      setStatus(_mlDataGrid.INDEX_STATUS.LOADED);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [dataViewFieldsData, dataViewFieldsError, dataViewFieldsIsError, dataViewFieldsIsLoading]);
  const dataViewFields = (0, _react.useMemo)(() => {
    let allPopulatedFields = Array.isArray(populatedFields) ? populatedFields : [];
    if (populatedFields === undefined && dataViewFieldsData) {
      // Get all field names for each returned doc and flatten it
      // to a list of unique field names used across all docs.
      const docs = dataViewFieldsData.hits.hits.map(d => {
        var _d$fields;
        return (0, _mlDataGrid.getProcessedFields)((_d$fields = d.fields) !== null && _d$fields !== void 0 ? _d$fields : {});
      });
      allPopulatedFields = [...new Set(docs.map(Object.keys).flat(1))];
    }
    const allDataViewFields = (0, _mlDataGrid.getFieldsFromKibanaDataView)(dataView);
    return allPopulatedFields.filter(d => allDataViewFields.includes(d)).sort();
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [dataViewFieldsData, populatedFields]);
  const columns = (0, _react.useMemo)(() => {
    if (typeof dataViewFields === 'undefined') {
      return [];
    }
    let result = [];

    // Get the the runtime fields that are defined from API field and index patterns
    if (combinedRuntimeMappings !== undefined) {
      result = Object.keys(combinedRuntimeMappings).map(fieldName => {
        const field = combinedRuntimeMappings[fieldName];
        const schema = (0, _mlDataGrid.getDataGridSchemaFromESFieldType)(field.type);
        return {
          id: fieldName,
          schema
        };
      });
    }

    // Combine the runtime field that are defined from API field
    dataViewFields.forEach(id => {
      const field = dataView.fields.getByName(id);
      if (!(field !== null && field !== void 0 && field.runtimeField)) {
        const schema = (0, _mlDataGrid.getDataGridSchemaFromKibanaFieldType)(field);
        result.push({
          id,
          schema
        });
      }
    });
    return result.sort((a, b) => a.id.localeCompare(b.id));
  }, [dataViewFields, dataView.fields, combinedRuntimeMappings]);

  // EuiDataGrid State

  const dataGrid = (0, _mlDataGrid.useDataGrid)(columns);
  const {
    chartsVisible,
    pagination,
    resetPagination,
    setColumnCharts,
    setCcsWarning,
    setErrorMessage,
    setRowCountInfo,
    setStatus,
    setTableItems,
    sortingColumns,
    tableItems
  } = dataGrid;
  (0, _react.useEffect)(() => {
    resetPagination();
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [JSON.stringify([query, timeRangeMs])]);
  const sort = sortingColumns.reduce((s, column) => {
    s[column.id] = {
      order: column.direction
    };
    return s;
  }, {});
  const {
    error: dataGridDataError,
    data: dataGridData,
    isError: dataGridDataIsError,
    isLoading: dataGridDataIsLoading
  } = (0, _use_data_search.useDataSearch)({
    index: indexPattern,
    body: {
      fields: ['*'],
      _source: false,
      query: (0, _mlQueryUtils.isDefaultQuery)(query) ? defaultQuery : queryWithBaseFilterCriteria,
      from: pagination.pageIndex * pagination.pageSize,
      size: pagination.pageSize,
      ...(Object.keys(sort).length > 0 ? {
        sort
      } : {}),
      ...((0, _mlRuntimeFieldUtils.isRuntimeMappings)(combinedRuntimeMappings) ? {
        runtime_mappings: combinedRuntimeMappings
      } : {})
    }
  },
  // Check whether fetching should be enabled
  dataViewFields !== undefined);
  (0, _react.useEffect)(() => {
    if (dataGridDataIsLoading && !dataGridDataIsError) {
      setErrorMessage('');
      setStatus(_mlDataGrid.INDEX_STATUS.LOADING);
    } else if (dataGridDataError !== null) {
      setErrorMessage((0, _errors.getErrorMessage)(dataGridDataError));
      setStatus(_mlDataGrid.INDEX_STATUS.ERROR);
    } else if (!dataGridDataIsLoading && !dataGridDataIsError && dataGridData !== undefined) {
      const isCrossClusterSearch = indexPattern.includes(':');
      const isMissingFields = dataGridData.hits.hits.every(d => typeof d.fields === 'undefined');
      const docs = dataGridData.hits.hits.map(d => {
        var _d$fields2;
        return (0, _mlDataGrid.getProcessedFields)((_d$fields2 = d.fields) !== null && _d$fields2 !== void 0 ? _d$fields2 : {});
      });
      setCcsWarning(isCrossClusterSearch && isMissingFields);
      setRowCountInfo({
        rowCount: typeof dataGridData.hits.total === 'number' ? dataGridData.hits.total : dataGridData.hits.total.value,
        rowCountRelation: typeof dataGridData.hits.total === 'number' ? 'eq' : dataGridData.hits.total.relation
      });
      setTableItems(docs);
      setStatus(_mlDataGrid.INDEX_STATUS.LOADED);
    }
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [dataGridDataError, dataGridDataIsError, dataGridDataIsLoading]);
  const allDataViewFieldNames = new Set(dataView.fields.map(f => f.name));
  const {
    error: histogramsForFieldsError,
    data: histogramsForFieldsData
  } = (0, _use_get_histograms_for_fields.useGetHistogramsForFields)(indexPattern, columns.filter(cT => dataGrid.visibleColumns.includes(cT.id)).map(cT => {
    // If a column field name has a corresponding keyword field,
    // fetch the keyword field instead to be able to do aggregations.
    const fieldName = cT.id;
    return (0, _field_utils.hasKeywordDuplicate)(fieldName, allDataViewFieldNames) ? {
      fieldName: `${fieldName}.keyword`,
      type: (0, _mlDataGrid.getFieldType)(undefined)
    } : {
      fieldName,
      type: (0, _mlDataGrid.getFieldType)(cT.schema)
    };
  }), (0, _mlQueryUtils.isDefaultQuery)(query) ? defaultQuery : queryWithBaseFilterCriteria, combinedRuntimeMappings, chartsVisible);
  (0, _react.useEffect)(() => {
    if (histogramsForFieldsError !== null) {
      (0, _mlDataGrid.showDataGridColumnChartErrorMessageToast)(histogramsForFieldsError, toastNotifications);
    }
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [histogramsForFieldsError]);
  (0, _react.useEffect)(() => {
    if (histogramsForFieldsData) {
      setColumnCharts(
      // revert field names with `.keyword` used to do aggregations to their original column name
      histogramsForFieldsData.map(d => ({
        ...d,
        ...((0, _field_utils.isKeywordDuplicate)(d.id, allDataViewFieldNames) ? {
          id: (0, _field_utils.removeKeywordPostfix)(d.id)
        } : {})
      })));
    }
    // custom comparison
    // eslint-disable-next-line react-hooks/exhaustive-deps
  }, [histogramsForFieldsData]);
  const renderCellValue = (0, _mlDataGrid.useRenderCellValue)(dataView, pagination, tableItems);
  if (dataGrid.status === _mlDataGrid.INDEX_STATUS.LOADED && dataViewFields !== undefined && Array.isArray(histogramsForFieldsData) && histogramsForFieldsData.length > 0 && loadIndexDataStartTime.current !== undefined) {
    const loadIndexDataDuration = window.performance.now() - loadIndexDataStartTime.current;

    // Set this to undefined so reporting the metric gets triggered only once.
    loadIndexDataStartTime.current = undefined;
    (0, _ebtTools.reportPerformanceMetricEvent)(analytics, {
      eventName: 'transformLoadIndexPreview',
      duration: loadIndexDataDuration
    });
  }
  return {
    ...dataGrid,
    renderCellValue
  };
};
exports.useIndexData = useIndexData;