"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerTelemetryOptInRoutes = registerTelemetryOptInRoutes;
var _rxjs = require("rxjs");
var _configSchema = require("@kbn/config-schema");
var _server = require("@kbn/core/server");
var _routes = require("../../common/routes");
var _telemetry_opt_in_stats = require("./telemetry_opt_in_stats");
var _saved_objects = require("../saved_objects");
var _telemetry_config = require("../telemetry_config");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0 and the Server Side Public License, v 1; you may not use this file except
 * in compliance with, at your election, the Elastic License 2.0 or the Server
 * Side Public License, v 1.
 */

function registerTelemetryOptInRoutes({
  config$,
  logger,
  router,
  currentKibanaVersion,
  telemetryCollectionManager
}) {
  const v2Handler = async (context, req, res) => {
    const newOptInStatus = req.body.enabled;
    const soClient = (await context.core).savedObjects.getClient({
      includedHiddenTypes: [_saved_objects.TELEMETRY_SAVED_OBJECT_TYPE]
    });
    const attributes = {
      enabled: newOptInStatus,
      lastVersionChecked: currentKibanaVersion
    };
    const config = await (0, _rxjs.firstValueFrom)(config$);
    let telemetrySavedObject;
    try {
      telemetrySavedObject = await (0, _saved_objects.getTelemetrySavedObject)(soClient);
    } catch (err) {
      if (_server.SavedObjectsErrorHelpers.isForbiddenError(err)) {
        // If we couldn't get the saved object due to lack of permissions,
        // we can assume the user won't be able to update it either
        return res.forbidden();
      }
    }
    const allowChangingOptInStatus = (0, _telemetry_config.getTelemetryAllowChangingOptInStatus)({
      configTelemetryAllowChangingOptInStatus: config.allowChangingOptInStatus,
      telemetrySavedObject
    });
    if (!allowChangingOptInStatus) {
      return res.badRequest({
        body: JSON.stringify({
          error: 'Not allowed to change Opt-in Status.'
        })
      });
    }
    const statsGetterConfig = {
      unencrypted: false
    };
    const optInStatus = await telemetryCollectionManager.getOptInStats(newOptInStatus, statsGetterConfig);
    if (config.sendUsageFrom === 'server') {
      const {
        appendServerlessChannelsSuffix,
        sendUsageTo
      } = config;
      (0, _telemetry_opt_in_stats.sendTelemetryOptInStatus)(telemetryCollectionManager, {
        appendServerlessChannelsSuffix,
        sendUsageTo,
        newOptInStatus,
        currentKibanaVersion
      }, statsGetterConfig).catch(err => {
        // The server is likely behind a firewall and can't reach the remote service
        logger.warn(`Failed to notify the telemetry endpoint about the opt-in selection. Possibly blocked by a firewall? - Error: ${err.message}`);
      });
    }
    try {
      await (0, _saved_objects.updateTelemetrySavedObject)(soClient, attributes);
    } catch (e) {
      if (_server.SavedObjectsErrorHelpers.isForbiddenError(e)) {
        return res.forbidden();
      }
    }
    const body = optInStatus;
    return res.ok({
      body
    });
  };
  const v2Validations = {
    request: {
      body: _configSchema.schema.object({
        enabled: _configSchema.schema.boolean()
      })
    },
    response: {
      200: {
        body: () => _configSchema.schema.arrayOf(_configSchema.schema.object({
          clusterUuid: _configSchema.schema.string(),
          stats: _configSchema.schema.string()
        }))
      }
    }
  };
  router.versioned.post({
    access: 'internal',
    path: _routes.OptInRoute
  })
  // Just because it used to be /v2/, we are creating identical v1 and v2.
  .addVersion({
    version: '1',
    validate: v2Validations
  }, v2Handler).addVersion({
    version: '2',
    validate: v2Validations
  }, v2Handler);
}