"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.normalizeYamlConfig = exports.getValueInSeconds = exports.getUnsupportedKeysError = exports.getOptionalListField = exports.getOptionalArrayField = exports.getNormalizeCommonFields = exports.getMonitorSchedule = exports.getMonitorLocations = exports.getMaxAttempts = exports.getInvalidUrlsOrHostsError = exports.getInvalidNamespaceError = exports.getHasTLSFields = exports.getCustomHeartbeatId = exports.flattenAndFormatObject = exports.LocationsMap = exports.InvalidLocationError = void 0;
var _lodash = require("lodash");
var _i18n = require("@kbn/i18n");
var _common = require("@kbn/fleet-plugin/common");
var _location_formatter = require("../../../../common/utils/location_formatter");
var _runtime_types = require("../../../../common/runtime_types");
var _monitor_defaults = require("../../../../common/constants/monitor_defaults");
var _private_formatters = require("../../formatters/private_formatters");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getNormalizeCommonFields = ({
  locations = [],
  privateLocations = [],
  monitor,
  projectId,
  namespace
}) => {
  var _monitor$enabled;
  const defaultFields = _monitor_defaults.DEFAULT_COMMON_FIELDS;
  const errors = [];
  if (monitor.namespace) {
    const namespaceError = (0, _common.isValidNamespace)(monitor.namespace).error;
    if (namespaceError) {
      errors.push(getInvalidNamespaceError(monitor, namespaceError));
    }
  }
  const monLocations = getMonitorLocations({
    monitorLocations: {
      locations: monitor.locations,
      privateLocations: monitor.privateLocations
    },
    allPrivateLocations: privateLocations,
    allPublicLocations: locations
  });
  const normalizedFields = {
    [_runtime_types.ConfigKey.JOURNEY_ID]: monitor.id || defaultFields[_runtime_types.ConfigKey.JOURNEY_ID],
    [_runtime_types.ConfigKey.MONITOR_SOURCE_TYPE]: _runtime_types.SourceType.PROJECT,
    [_runtime_types.ConfigKey.NAME]: monitor.name || '',
    [_runtime_types.ConfigKey.SCHEDULE]: {
      number: `${monitor.schedule}`,
      unit: _runtime_types.ScheduleUnit.MINUTES
    },
    [_runtime_types.ConfigKey.PROJECT_ID]: projectId,
    [_runtime_types.ConfigKey.LOCATIONS]: monLocations,
    [_runtime_types.ConfigKey.TAGS]: getOptionalListField(monitor.tags) || defaultFields[_runtime_types.ConfigKey.TAGS],
    [_runtime_types.ConfigKey.NAMESPACE]: monitor.namespace || (0, _private_formatters.formatKibanaNamespace)(namespace) || defaultFields[_runtime_types.ConfigKey.NAMESPACE],
    [_runtime_types.ConfigKey.ORIGINAL_SPACE]: namespace || defaultFields[_runtime_types.ConfigKey.NAMESPACE],
    [_runtime_types.ConfigKey.CUSTOM_HEARTBEAT_ID]: getCustomHeartbeatId(monitor, projectId, namespace),
    [_runtime_types.ConfigKey.ENABLED]: (_monitor$enabled = monitor.enabled) !== null && _monitor$enabled !== void 0 ? _monitor$enabled : defaultFields[_runtime_types.ConfigKey.ENABLED],
    [_runtime_types.ConfigKey.TIMEOUT]: monitor.timeout ? getValueInSeconds(monitor.timeout) : defaultFields[_runtime_types.ConfigKey.TIMEOUT],
    [_runtime_types.ConfigKey.CONFIG_HASH]: monitor.hash || defaultFields[_runtime_types.ConfigKey.CONFIG_HASH],
    [_runtime_types.ConfigKey.MAX_ATTEMPTS]: getMaxAttempts(monitor.retestOnFailure),
    [_runtime_types.ConfigKey.PARAMS]: Object.keys(monitor.params || {}).length ? JSON.stringify(monitor.params) : defaultFields[_runtime_types.ConfigKey.PARAMS],
    // picking out keys specifically, so users can't add arbitrary fields
    [_runtime_types.ConfigKey.ALERT_CONFIG]: getAlertConfig(monitor)
  };
  return {
    normalizedFields,
    errors
  };
};
exports.getNormalizeCommonFields = getNormalizeCommonFields;
const getAlertConfig = monitor => {
  var _defaultFields$Config, _ref, _monitor$alert$status, _monitor$alert, _monitor$alert$status2, _defaultFields$Config2, _defaultFields$Config3, _defaultFields$Config4, _ref2, _monitor$alert$tls$en, _monitor$alert2, _monitor$alert2$tls, _defaultFields$Config5, _defaultFields$Config6;
  const defaultFields = _monitor_defaults.DEFAULT_COMMON_FIELDS;
  return monitor.alert ? {
    ...defaultFields[_runtime_types.ConfigKey.ALERT_CONFIG],
    status: {
      ...((_defaultFields$Config = defaultFields[_runtime_types.ConfigKey.ALERT_CONFIG]) === null || _defaultFields$Config === void 0 ? void 0 : _defaultFields$Config.status),
      enabled: (_ref = (_monitor$alert$status = (_monitor$alert = monitor.alert) === null || _monitor$alert === void 0 ? void 0 : (_monitor$alert$status2 = _monitor$alert.status) === null || _monitor$alert$status2 === void 0 ? void 0 : _monitor$alert$status2.enabled) !== null && _monitor$alert$status !== void 0 ? _monitor$alert$status : (_defaultFields$Config2 = defaultFields[_runtime_types.ConfigKey.ALERT_CONFIG]) === null || _defaultFields$Config2 === void 0 ? void 0 : (_defaultFields$Config3 = _defaultFields$Config2.status) === null || _defaultFields$Config3 === void 0 ? void 0 : _defaultFields$Config3.enabled) !== null && _ref !== void 0 ? _ref : true
    },
    tls: {
      ...((_defaultFields$Config4 = defaultFields[_runtime_types.ConfigKey.ALERT_CONFIG]) === null || _defaultFields$Config4 === void 0 ? void 0 : _defaultFields$Config4.tls),
      enabled: (_ref2 = (_monitor$alert$tls$en = (_monitor$alert2 = monitor.alert) === null || _monitor$alert2 === void 0 ? void 0 : (_monitor$alert2$tls = _monitor$alert2.tls) === null || _monitor$alert2$tls === void 0 ? void 0 : _monitor$alert2$tls.enabled) !== null && _monitor$alert$tls$en !== void 0 ? _monitor$alert$tls$en : (_defaultFields$Config5 = defaultFields[_runtime_types.ConfigKey.ALERT_CONFIG]) === null || _defaultFields$Config5 === void 0 ? void 0 : (_defaultFields$Config6 = _defaultFields$Config5.tls) === null || _defaultFields$Config6 === void 0 ? void 0 : _defaultFields$Config6.enabled) !== null && _ref2 !== void 0 ? _ref2 : true
    }
  } : defaultFields[_runtime_types.ConfigKey.ALERT_CONFIG];
};
const ONLY_ONE_ATTEMPT = 1;
const getMaxAttempts = retestOnFailure => {
  const defaultFields = _monitor_defaults.DEFAULT_COMMON_FIELDS;
  if (retestOnFailure) {
    return defaultFields[_runtime_types.ConfigKey.MAX_ATTEMPTS];
  } else if (retestOnFailure === false) {
    return ONLY_ONE_ATTEMPT;
  }
  return defaultFields[_runtime_types.ConfigKey.MAX_ATTEMPTS];
};
exports.getMaxAttempts = getMaxAttempts;
const getCustomHeartbeatId = (monitor, projectId, namespace) => {
  return `${monitor.id}-${projectId}-${namespace}`;
};
exports.getCustomHeartbeatId = getCustomHeartbeatId;
const getMonitorSchedule = schedule => {
  if (typeof schedule === 'number' || typeof schedule === 'string') {
    return {
      number: `${schedule}`,
      unit: _runtime_types.ScheduleUnit.MINUTES
    };
  }
  return schedule;
};
exports.getMonitorSchedule = getMonitorSchedule;
const LocationsMap = exports.LocationsMap = {
  japan: 'asia-northeast1-a',
  india: 'asia-south1-a',
  singapore: 'asia-southeast1-a',
  australia_east: 'australia-southeast1-a',
  united_kingdom: 'europe-west2-a',
  germany: 'europe-west3-a',
  canada_east: 'northamerica-northeast1-a',
  brazil: 'southamerica-east1-a',
  us_east: 'us-east4-a',
  us_west: 'us-west1-a'
};
const getMonitorLocations = ({
  allPrivateLocations,
  allPublicLocations,
  monitorLocations
}) => {
  var _monitorLocations$loc, _monitorLocations$pri;
  const invalidPublicLocations = [];
  const invalidPrivateLocations = [];
  const publicLocs = ((_monitorLocations$loc = monitorLocations.locations) === null || _monitorLocations$loc === void 0 ? void 0 : _monitorLocations$loc.map(locationId => {
    const locationFound = allPublicLocations.find(location => location.id === (LocationsMap[locationId] || locationId) || location.id === locationId);
    if (locationFound) {
      return locationFound;
    } else {
      invalidPublicLocations.push(locationId);
    }
  })) || [];
  const privateLocs = ((_monitorLocations$pri = monitorLocations.privateLocations) === null || _monitorLocations$pri === void 0 ? void 0 : _monitorLocations$pri.map(locationName => {
    const locationFound = allPrivateLocations.find(location => location.label.toLowerCase() === locationName.toLowerCase() || location.id.toLowerCase() === locationName.toLowerCase());
    if (locationFound) {
      return locationFound;
    } else {
      invalidPrivateLocations.push(locationName);
    }
  })) || [];
  if (invalidPublicLocations.length || invalidPrivateLocations.length) {
    throw new InvalidLocationError(getInvalidLocationError(invalidPublicLocations, invalidPrivateLocations, allPublicLocations, allPrivateLocations));
  }
  const allLocations = [...publicLocs, ...privateLocs].filter(location => location !== undefined).map(loc => (0, _location_formatter.formatLocation)(loc));

  // return only unique locations
  return (0, _lodash.uniqBy)(allLocations, 'id');
};
exports.getMonitorLocations = getMonitorLocations;
class InvalidLocationError extends Error {
  constructor(message) {
    super(message);
    this.name = 'InvalidLocationError';
  }
}
exports.InvalidLocationError = InvalidLocationError;
const UNSUPPORTED_OPTION_TITLE = _i18n.i18n.translate('xpack.synthetics.projectMonitorApi.validation.unsupportedOption.title', {
  defaultMessage: 'Unsupported Heartbeat option'
});
const INVALID_CONFIGURATION_TITLE = _i18n.i18n.translate('xpack.synthetics.projectMonitorApi.validation.invalidConfiguration.title', {
  defaultMessage: 'Invalid Heartbeat configuration'
});
const INVALID_NAMESPACE_TITLE = _i18n.i18n.translate('xpack.synthetics.projectMonitorApi.validation.invalidNamespace.title', {
  defaultMessage: 'Invalid namespace'
});
const getUnsupportedKeysError = (monitor, unsupportedKeys, version) => ({
  id: monitor.id,
  reason: UNSUPPORTED_OPTION_TITLE,
  details: `The following Heartbeat options are not supported for ${monitor.type} project monitors in ${version}: ${unsupportedKeys.join('|')}. You monitor was not created or updated.`
});
exports.getUnsupportedKeysError = getUnsupportedKeysError;
const getInvalidUrlsOrHostsError = (monitor, key, version) => ({
  id: monitor.id,
  reason: INVALID_CONFIGURATION_TITLE,
  details: _i18n.i18n.translate('xpack.synthetics.projectMonitorApi.validation.invalidUrlOrHosts.description', {
    defaultMessage: '`{monitorType}` project monitors must have exactly one value for field `{key}` in version `{version}`. Your monitor was not created or updated.',
    values: {
      monitorType: monitor.type,
      key,
      version
    }
  })
});
exports.getInvalidUrlsOrHostsError = getInvalidUrlsOrHostsError;
const getInvalidLocationError = (invalidPublic, invalidPrivate, allPublicLocations, allPrivateLocations) => {
  const availablePublicMsg = allPublicLocations.length === 0 ? 'No Elastic managed location available to use.' : `Available locations are '${allPublicLocations.map(l => l.id).join('|')}'`;
  const availablePrivateMsg = allPrivateLocations.length === 0 ? 'No private location available to use.' : `Available private locations are '${allPrivateLocations.map(l => l.label).join('|')}'`;
  return _i18n.i18n.translate('xpack.synthetics.projectMonitorApi.validation.invalidLocations', {
    defaultMessage: 'Invalid locations specified.{invalidPublicLocation}{invalidPrivateLocation}',
    values: {
      invalidPublicLocation: invalidPublic.length > 0 ? ` Elastic managed Location(s) '${invalidPublic.join('|')}' not found. ${availablePublicMsg}` : '',
      invalidPrivateLocation: invalidPrivate.length > 0 ? ` Private Location(s) '${invalidPrivate.join('|')}' not found. ${availablePrivateMsg}` : ''
    }
  });
};
const getInvalidNamespaceError = (monitor, error) => ({
  id: monitor.id,
  reason: INVALID_NAMESPACE_TITLE,
  details: error
});
exports.getInvalidNamespaceError = getInvalidNamespaceError;
const getValueInSeconds = value => {
  const keyMap = {
    h: 60 * 60,
    m: 60,
    s: 1
  };
  const key = value.slice(-1);
  const time = parseInt(value.slice(0, -1), 10);
  const valueInSeconds = time * (keyMap[key] || 1);
  return typeof valueInSeconds === 'number' ? `${valueInSeconds}` : null;
};

/**
 * Accounts for array values that are optionally defined as a comma seperated list
 *
 * @param {Array | string} [value]
 * @returns {array} Returns an array
 */
exports.getValueInSeconds = getValueInSeconds;
const getOptionalListField = value => {
  if (Array.isArray(value)) {
    return value;
  }
  return value ? value.split(',') : [];
};

/**
 * Accounts for heartbeat fields that are optionally an array or single string
 *
 * @param {Array | string} [value]
 * @returns {string} Returns first item when the value is an array, or the value itself
 */
exports.getOptionalListField = getOptionalListField;
const getOptionalArrayField = (value = '') => {
  const array = getOptionalListField(value);
  return array[0];
};

/**
 * Flattens arbitrary yaml into a synthetics monitor compatible configuration
 *
 * @param {Object} [monitor]
 * @returns {Object} Returns an object containing synthetics-compatible configuration keys
 */
exports.getOptionalArrayField = getOptionalArrayField;
const flattenAndFormatObject = (obj, prefix = '', keys) => Object.keys(obj).reduce((acc, k) => {
  const pre = prefix.length ? prefix + '.' : '';
  const key = pre + k;

  /* If the key is an array of numbers, convert to an array of strings */
  if (Array.isArray(obj[k])) {
    acc[key] = obj[k].map(value => typeof value === 'number' ? String(value) : value);
    return acc;
  }

  /* if the key is a supported key stop flattening early */
  if (keys.includes(key)) {
    acc[key] = obj[k];
    return acc;
  }
  if (typeof obj[k] === 'object') {
    Object.assign(acc, flattenAndFormatObject(obj[k], pre + k, keys));
  } else {
    acc[key] = obj[k];
  }
  return acc;
}, {});
exports.flattenAndFormatObject = flattenAndFormatObject;
const normalizeYamlConfig = monitor => {
  const defaultFields = _monitor_defaults.DEFAULT_FIELDS[monitor.type];
  const supportedKeys = Object.keys(defaultFields);
  const flattenedConfig = flattenAndFormatObject(monitor, '', supportedKeys);
  const {
    locations: _locations,
    privateLocations: _privateLocations,
    content: _content,
    id: _id,
    retestOnFailure: _retestOnFailure,
    ...yamlConfig
  } = flattenedConfig;
  const unsupportedKeys = Object.keys(yamlConfig).filter(key => !supportedKeys.includes(key));
  const supportedYamlConfig = (0, _lodash.omit)(yamlConfig, unsupportedKeys);
  return {
    yamlConfig: supportedYamlConfig,
    unsupportedKeys
  };
};

// returns true when any ssl fields are defined
exports.normalizeYamlConfig = normalizeYamlConfig;
const getHasTLSFields = monitor => Object.keys(monitor).some(key => key.includes('ssl'));
exports.getHasTLSFields = getHasTLSFields;