"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.isPartialCreateAlertSchema = exports.decodeCreateAlert = void 0;
var rt = _interopRequireWildcard(require("io-ts"));
var _lodash = require("lodash");
var _schema_utils = require("../schema_utils");
var i18n = _interopRequireWildcard(require("./translations"));
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && {}.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const MessageNonEmptyString = new rt.Type('MessageNonEmptyString', rt.string.is, (input, context) => {
  if (input === undefined) {
    return rt.failure(input, context, i18n.MESSAGE_NOT_DEFINED);
  } else if (typeof input !== 'string') {
    return rt.failure(input, context);
  } else if ((0, _lodash.isEmpty)(input.trim())) {
    return rt.failure(input, context, i18n.MESSAGE_NON_WHITESPACE);
  } else {
    return rt.success(input);
  }
}, rt.identity);
const ResponderTypes = rt.union([rt.literal('team'), rt.literal('user'), rt.literal('escalation'), rt.literal('schedule')]);
const CreateAlertSchemaOptionalProps = rt.partial(rt.type({
  alias: rt.string,
  description: rt.string,
  responders: rt.array(rt.union([rt.strict({
    name: rt.string,
    type: ResponderTypes
  }), rt.strict({
    id: rt.string,
    type: ResponderTypes
  }), rt.strict({
    username: rt.string,
    type: rt.literal('user')
  })])),
  visibleTo: rt.array(rt.union([rt.strict({
    name: rt.string,
    type: rt.literal('team')
  }), rt.strict({
    id: rt.string,
    type: rt.literal('team')
  }), rt.strict({
    id: rt.string,
    type: rt.literal('user')
  }), rt.strict({
    username: rt.string,
    type: rt.literal('user')
  })])),
  actions: rt.array(rt.string),
  tags: rt.array(rt.string),
  details: rt.record(rt.string, rt.string),
  entity: rt.string,
  source: rt.string,
  priority: rt.union([rt.literal('P1'), rt.literal('P2'), rt.literal('P3'), rt.literal('P4'), rt.literal('P5')]),
  user: rt.string,
  note: rt.string
}).props);

/**
 * This schema is duplicated from the server. The only difference is that it is using io-ts vs kbn-schema.
 * NOTE: This schema must be the same as defined here: x-pack/plugins/stack_connectors/server/connector_types/stack/opsgenie/schema.ts
 *
 * The reason it is duplicated here is because the server uses kbn-schema which uses Joi under the hood. If we import
 * Joi on the frontend it will cause ~500KB of data to be loaded on page loads. To avoid this we'll use io-ts in the frontend.
 * Ideally we could use io-ts in the backend as well but the server requires kbn-schema to be used.
 *
 * Issue: https://github.com/elastic/kibana/issues/143891
 *
 * For more information on the Opsgenie create alert schema see: https://docs.opsgenie.com/docs/alert-api#create-alert
 */
const CreateAlertSchema = rt.intersection([rt.strict({
  message: MessageNonEmptyString
}), rt.exact(CreateAlertSchemaOptionalProps)]);
/**
 * This schema should match CreateAlertSchema except that all fields are optional and message is only enforced as a string.
 * Enforcing message as only a string accommodates the following scenario:
 *
 * If a user deletes an action in the rule form at index 0, and the
 * action at index 1 had the message field specified with all spaces, the message field is technically invalid but
 * we want to allow it to pass the partial check so that the form is still populated with the invalid value. Otherwise the
 * forum will be reset and the user would lose the information (although it is invalid) they had entered
 */
const PartialCreateAlertSchema = rt.exact(rt.intersection([rt.partial(rt.type({
  message: rt.string
}).props), CreateAlertSchemaOptionalProps]));
const isPartialCreateAlertSchema = data => {
  try {
    (0, _schema_utils.decodeSchema)(PartialCreateAlertSchema, data);
    return true;
  } catch (error) {
    return false;
  }
};
exports.isPartialCreateAlertSchema = isPartialCreateAlertSchema;
const decodeCreateAlert = data => {
  return (0, _schema_utils.decodeSchema)(CreateAlertSchema, data);
};
exports.decodeCreateAlert = decodeCreateAlert;