"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fetchSearchSourceQuery = fetchSearchSourceQuery;
exports.generateLink = generateLink;
exports.getSmallerDataViewSpec = getSmallerDataViewSpec;
exports.updateFilterReferences = updateFilterReferences;
exports.updateSearchSource = updateSearchSource;
var _esQuery = require("@kbn/es-query");
var _common = require("@kbn/data-plugin/common");
var _common2 = require("@kbn/triggers-actions-ui-plugin/common");
var _common3 = require("../../../../common");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

async function fetchSearchSourceQuery({
  ruleId,
  alertLimit,
  params,
  latestTimestamp,
  spacePrefix,
  services,
  dateStart,
  dateEnd
}) {
  const {
    logger,
    searchSourceClient
  } = services;
  const isGroupAgg = (0, _common2.isGroupAggregation)(params.termField);
  const isCountAgg = (0, _common2.isCountAggregation)(params.aggType);
  const initialSearchSource = await searchSourceClient.create(params.searchConfiguration);
  const index = initialSearchSource.getField('index');
  const {
    searchSource,
    filterToExcludeHitsFromPreviousRun
  } = updateSearchSource(initialSearchSource, index, params, latestTimestamp, dateStart, dateEnd, alertLimit);
  logger.debug(`search source query rule (${ruleId}) query: ${JSON.stringify(searchSource.getSearchRequestBody())}`);
  const searchResult = await searchSource.fetch();
  const link = await generateLink(initialSearchSource, services.share.url.locators.get('DISCOVER_APP_LOCATOR'), services.dataViews, index, dateStart, dateEnd, spacePrefix, filterToExcludeHitsFromPreviousRun);
  return {
    link,
    numMatches: Number(searchResult.hits.total),
    searchResult,
    parsedResults: (0, _common2.parseAggregationResults)({
      isCountAgg,
      isGroupAgg,
      esResult: searchResult,
      sourceFieldsParams: params.sourceFields
    }),
    index: [index.name]
  };
}
function updateSearchSource(searchSource, index, params, latestTimestamp, dateStart, dateEnd, alertLimit) {
  const isGroupAgg = (0, _common2.isGroupAggregation)(params.termField);
  const timeField = index.getTimeField();
  if (!timeField) {
    throw new Error(`Data view with ID ${index.id} no longer contains a time field.`);
  }
  searchSource.setField('size', isGroupAgg ? 0 : params.size);
  const filters = [(0, _esQuery.buildRangeFilter)(timeField, {
    lte: dateEnd,
    gte: dateStart,
    format: 'strict_date_optional_time'
  }, index)];
  let filterToExcludeHitsFromPreviousRun = null;
  if (params.excludeHitsFromPreviousRun) {
    if (latestTimestamp && latestTimestamp > dateStart) {
      // add additional filter for documents with a timestamp greater than
      // the timestamp of the previous run, so that those documents are not counted twice
      filterToExcludeHitsFromPreviousRun = (0, _esQuery.buildRangeFilter)(timeField, {
        gt: latestTimestamp,
        format: 'strict_date_optional_time'
      }, index);
      filters.push(filterToExcludeHitsFromPreviousRun);
    }
  }
  const searchSourceChild = searchSource.createChild();
  if (!isGroupAgg) {
    searchSourceChild.setField('trackTotalHits', true);
  }
  searchSourceChild.setField('filter', filters);
  searchSourceChild.setField('sort', [{
    [timeField.name]: {
      order: _common.SortDirection.desc,
      format: 'strict_date_optional_time||epoch_millis'
    }
  }]);
  searchSourceChild.setField('aggs', (0, _common2.buildAggregation)({
    aggType: params.aggType,
    aggField: params.aggField,
    termField: params.termField,
    termSize: params.termSize,
    sourceFieldsParams: params.sourceFields,
    condition: {
      resultLimit: alertLimit,
      conditionScript: (0, _common3.getComparatorScript)(params.thresholdComparator, params.threshold, _common2.BUCKET_SELECTOR_FIELD)
    },
    ...(isGroupAgg ? {
      topHitsSize: params.size
    } : {})
  }));
  return {
    searchSource: searchSourceChild,
    filterToExcludeHitsFromPreviousRun
  };
}
async function generateLink(searchSource, discoverLocator, dataViews, dataViewToUpdate, dateStart, dateEnd, spacePrefix, filterToExcludeHitsFromPreviousRun) {
  const prevFilters = [...(searchSource.getField('filter') || [])];
  if (filterToExcludeHitsFromPreviousRun) {
    // Using the same additional filter as in the alert check above.
    // We cannot simply pass `latestTimestamp` to `timeRange.from` Discover locator params
    // as that would include `latestTimestamp` itself in the Discover results which would be wrong.
    // Results should be after `latestTimestamp` and within `dateStart` and `dateEnd`.
    prevFilters.push(filterToExcludeHitsFromPreviousRun);
  }

  // make new adhoc data view
  const newDataView = await dataViews.create({
    ...dataViewToUpdate.toSpec(false),
    version: undefined,
    id: undefined
  }, true);
  const updatedFilters = updateFilterReferences(prevFilters, dataViewToUpdate.id, newDataView.id);
  const redirectUrlParams = {
    dataViewSpec: getSmallerDataViewSpec(newDataView),
    filters: updatedFilters,
    query: searchSource.getField('query'),
    timeRange: {
      from: dateStart,
      to: dateEnd
    },
    isAlertResults: true
  };

  // use `lzCompress` flag for making the link readable during debugging/testing
  // const redirectUrl = discoverLocator!.getRedirectUrl(redirectUrlParams, { lzCompress: false });
  const redirectUrl = discoverLocator.getRedirectUrl(redirectUrlParams);
  const [start, end] = redirectUrl.split('/app');
  return start + spacePrefix + '/app' + end;
}
function updateFilterReferences(filters, fromDataView, toDataView) {
  return (filters || []).map(filter => {
    if (filter.meta.index === fromDataView) {
      return {
        ...filter,
        meta: {
          ...filter.meta,
          index: toDataView
        }
      };
    } else {
      return filter;
    }
  });
}
function getSmallerDataViewSpec(dataView) {
  return dataView.toMinimalSpec({
    keepFieldAttrs: ['customLabel']
  });
}