"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.fromRemoteSummaryDocumentToSloDefinition = fromRemoteSummaryDocumentToSloDefinition;
var _securitysolutionIoTsUtils = require("@kbn/securitysolution-io-ts-utils");
var _sloSchema = require("@kbn/slo-schema");
var _std = require("@kbn/std");
var _Either = require("fp-ts/lib/Either");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function fromRemoteSummaryDocumentToSloDefinition(summaryDoc, logger) {
  var _summaryDoc$slo$objec, _summaryDoc$slo$objec2, _summaryDoc$slo$creat, _summaryDoc$slo$updat;
  const res = _sloSchema.sloDefinitionSchema.decode({
    id: summaryDoc.slo.id,
    name: summaryDoc.slo.name,
    description: summaryDoc.slo.description,
    indicator: getIndicator(summaryDoc, logger),
    timeWindow: summaryDoc.slo.timeWindow,
    budgetingMethod: summaryDoc.slo.budgetingMethod,
    objective: {
      target: summaryDoc.slo.objective.target,
      timesliceTarget: (_summaryDoc$slo$objec = summaryDoc.slo.objective.timesliceTarget) !== null && _summaryDoc$slo$objec !== void 0 ? _summaryDoc$slo$objec : undefined,
      timesliceWindow: (_summaryDoc$slo$objec2 = summaryDoc.slo.objective.timesliceWindow) !== null && _summaryDoc$slo$objec2 !== void 0 ? _summaryDoc$slo$objec2 : undefined
    },
    settings: {
      syncDelay: '1m',
      frequency: '1m'
    },
    revision: summaryDoc.slo.revision,
    enabled: true,
    tags: summaryDoc.slo.tags,
    createdAt: (_summaryDoc$slo$creat = summaryDoc.slo.createdAt) !== null && _summaryDoc$slo$creat !== void 0 ? _summaryDoc$slo$creat : '2024-01-01T00:00:00.000Z',
    // fallback prior 8.14
    updatedAt: (_summaryDoc$slo$updat = summaryDoc.slo.updatedAt) !== null && _summaryDoc$slo$updat !== void 0 ? _summaryDoc$slo$updat : '2024-01-01T00:00:00.000Z',
    // fallback prior 8.14
    groupBy: summaryDoc.slo.groupBy,
    version: 1
  });
  if ((0, _Either.isLeft)(res)) {
    const errors = (0, _securitysolutionIoTsUtils.formatErrors)(res.left);
    logger.error(`Invalid remote stored summary SLO with id [${summaryDoc.slo.id}]`);
    logger.error(errors.join('|'));
    return undefined;
  }
  return res.right;
}

/**
 * Temporary documents priors to 8.14 don't have indicator.params, therefore we need to fallback to a dummy
 */
function getIndicator(summaryDoc, logger) {
  const res = _sloSchema.indicatorSchema.decode(summaryDoc.slo.indicator);
  if ((0, _Either.isLeft)(res)) {
    const errors = (0, _securitysolutionIoTsUtils.formatErrors)(res.left);
    logger.info(`Invalid indicator from remote summary SLO id [${summaryDoc.slo.id}] - Fallback on dummy indicator`);
    logger.info(errors.join('|'));
    return getDummyIndicator(summaryDoc);
  }
  return res.right;
}
function getDummyIndicator(summaryDoc) {
  const indicatorType = summaryDoc.slo.indicator.type;
  let indicator;
  switch (indicatorType) {
    case 'sli.kql.custom':
      indicator = {
        type: indicatorType,
        params: {
          index: '',
          good: '',
          total: '',
          timestampField: ''
        }
      };
      break;
    case 'sli.apm.transactionDuration':
      indicator = {
        type: indicatorType,
        params: {
          environment: '',
          service: '',
          transactionType: '',
          transactionName: '',
          threshold: 0,
          index: ''
        }
      };
      break;
    case 'sli.apm.transactionErrorRate':
      indicator = {
        type: indicatorType,
        params: {
          environment: '',
          service: '',
          transactionType: '',
          transactionName: '',
          index: ''
        }
      };
      break;
    case 'sli.metric.custom':
      indicator = {
        type: indicatorType,
        params: {
          index: '',
          good: {
            metrics: [{
              name: '',
              aggregation: 'sum',
              field: ''
            }],
            equation: ''
          },
          total: {
            metrics: [{
              name: '',
              aggregation: 'sum',
              field: ''
            }],
            equation: ''
          },
          timestampField: ''
        }
      };
      break;
    case 'sli.metric.timeslice':
      indicator = {
        type: indicatorType,
        params: {
          index: '',
          metric: {
            metrics: [],
            equation: '',
            threshold: 0,
            comparator: 'GT'
          },
          timestampField: ''
        }
      };
      break;
    case 'sli.histogram.custom':
      indicator = {
        type: indicatorType,
        params: {
          index: '',
          timestampField: '',
          good: {
            field: '',
            aggregation: 'value_count'
          },
          total: {
            field: '',
            aggregation: 'value_count'
          }
        }
      };
      break;
    case 'sli.synthetics.availability':
      indicator = {
        type: indicatorType,
        params: {
          projects: [],
          tags: [],
          monitorIds: [{
            value: '*',
            label: 'All'
          }],
          index: 'synthetics-*',
          filter: ''
        }
      };
      break;
    default:
      (0, _std.assertNever)(indicatorType);
  }
  return indicator;
}