"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getDeleteQueryFilter = exports.TASK_TYPE = exports.SloOrphanSummaryCleanupTask = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _constants = require("@kbn/spaces-plugin/common/constants");
var _saved_objects = require("../../saved_objects");
var _constants2 = require("../../../common/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const TASK_TYPE = exports.TASK_TYPE = 'SLO:ORPHAN_SUMMARIES-CLEANUP-TASK';
const getDeleteQueryFilter = sloSummaryIdsToDelete => {
  return sloSummaryIdsToDelete.map(({
    id,
    revision
  }) => {
    return {
      bool: {
        must: [{
          term: {
            'slo.id': id
          }
        }, {
          term: {
            'slo.revision': Number(revision)
          }
        }]
      }
    };
  });
};
exports.getDeleteQueryFilter = getDeleteQueryFilter;
class SloOrphanSummaryCleanupTask {
  constructor(taskManager, logger, config) {
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "taskManager", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "esClient", void 0);
    (0, _defineProperty2.default)(this, "config", void 0);
    (0, _defineProperty2.default)(this, "runTask", async () => {
      if (this.soClient && this.esClient) {
        let searchAfterKey;
        do {
          const {
            sloSummaryIds,
            searchAfter
          } = await this.fetchSloSummariesIds(searchAfterKey);
          if (sloSummaryIds.length === 0) {
            return;
          }
          searchAfterKey = searchAfter;
          const ids = sloSummaryIds.map(({
            id
          }) => id);
          const sloDefinitions = await this.findSloDefinitions(ids);
          const sloSummaryIdsToDelete = sloSummaryIds.filter(({
            id,
            revision
          }) => !sloDefinitions.find(attributes => attributes.id === id && attributes.revision === revision));
          if (sloSummaryIdsToDelete.length > 0) {
            this.logger.info(`[SLO] Deleting ${sloSummaryIdsToDelete.length} SLO Summaries from the summary index`);
            await this.esClient.deleteByQuery({
              wait_for_completion: false,
              index: _constants2.SLO_SUMMARY_DESTINATION_INDEX_PATTERN,
              query: {
                bool: {
                  should: getDeleteQueryFilter(sloSummaryIdsToDelete.sort())
                }
              }
            });
          }
        } while (searchAfterKey);
      }
    });
    (0, _defineProperty2.default)(this, "fetchSloSummariesIds", async searchAfter => {
      var _result$aggregations$, _result$aggregations, _result$aggregations2;
      this.logger.debug(`[SLO] Fetching SLO Summaries ids after ${searchAfter}`);
      if (!this.esClient) {
        return {
          searchAfter: undefined,
          sloSummaryIds: []
        };
      }
      const size = 1000;
      const result = await this.esClient.search({
        size: 0,
        index: _constants2.SLO_SUMMARY_DESTINATION_INDEX_PATTERN,
        aggs: {
          slos: {
            composite: {
              size,
              sources: [{
                id: {
                  terms: {
                    field: 'slo.id'
                  }
                }
              }, {
                revision: {
                  terms: {
                    field: 'slo.revision'
                  }
                }
              }],
              after: searchAfter
            }
          }
        }
      });
      const aggBuckets = (_result$aggregations$ = (_result$aggregations = result.aggregations) === null || _result$aggregations === void 0 ? void 0 : _result$aggregations.slos.buckets) !== null && _result$aggregations$ !== void 0 ? _result$aggregations$ : [];
      if (aggBuckets.length === 0) {
        return {
          searchAfter: undefined,
          sloSummaryIds: []
        };
      }
      const newSearchAfter = aggBuckets.length < size ? undefined : (_result$aggregations2 = result.aggregations) === null || _result$aggregations2 === void 0 ? void 0 : _result$aggregations2.slos.after_key;
      const sloSummaryIds = aggBuckets.map(({
        key
      }) => {
        return {
          id: String(key.id),
          revision: Number(key.revision)
        };
      });
      return {
        searchAfter: newSearchAfter,
        sloSummaryIds
      };
    });
    (0, _defineProperty2.default)(this, "findSloDefinitions", async ids => {
      var _this$soClient, _sloDefinitions$saved;
      const sloDefinitions = await ((_this$soClient = this.soClient) === null || _this$soClient === void 0 ? void 0 : _this$soClient.find({
        type: _saved_objects.SO_SLO_TYPE,
        page: 1,
        perPage: ids.length,
        filter: `slo.attributes.id:(${ids.join(' or ')})`,
        namespaces: [_constants.ALL_SPACES_ID],
        fields: ['id', 'revision']
      }));
      return (_sloDefinitions$saved = sloDefinitions === null || sloDefinitions === void 0 ? void 0 : sloDefinitions.saved_objects.map(({
        attributes
      }) => attributes)) !== null && _sloDefinitions$saved !== void 0 ? _sloDefinitions$saved : [];
    });
    this.logger = logger;
    this.config = config;
    taskManager.registerTaskDefinitions({
      [TASK_TYPE]: {
        title: 'SLO Definitions Cleanup Task',
        timeout: '3m',
        maxAttempts: 1,
        createTaskRunner: ({
          taskInstance
        }) => {
          return {
            run: async () => {
              return this.runTask();
            },
            cancel: async () => {
              this.abortController.abort('[SLO] Definitions clean up Task timed out');
            }
          };
        }
      }
    });
  }
  get taskId() {
    return `${TASK_TYPE}:1.0.0`;
  }
  async start(taskManager, soClient, esClient) {
    this.taskManager = taskManager;
    this.soClient = soClient;
    this.esClient = esClient;
    if (!taskManager) {
      this.logger.info('[SLO] Missing required service during startup, skipping task.');
      return;
    }
    if (this.config.sloOrphanSummaryCleanUpTaskEnabled) {
      this.taskManager.ensureScheduled({
        id: this.taskId,
        taskType: TASK_TYPE,
        schedule: {
          interval: '1h'
        },
        scope: ['observability', 'slo'],
        state: {},
        params: {}
      });
    } else {
      this.taskManager.removeIfExists(this.taskId);
    }
  }
}
exports.SloOrphanSummaryCleanupTask = SloOrphanSummaryCleanupTask;