"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultSummaryClient = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _constants = require("../../common/constants");
var _services = require("../domain/services");
var _date_range = require("../domain/services/date_range");
var _utils = require("./utils");
var _compute_total_slices_from_date_range = require("./utils/compute_total_slices_from_date_range");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

// This is called "SummaryClient" but is responsible for:
// - computing summary
// - formatting groupings
// - adding extra Meta parameter for synthetics

class DefaultSummaryClient {
  constructor(esClient) {
    this.esClient = esClient;
  }
  async computeSummary({
    slo,
    instanceId,
    remoteName
  }) {
    var _result$aggregations$, _result$aggregations, _result$aggregations$2, _result$aggregations$3, _result$aggregations2, _result$aggregations3, _result$aggregations4, _result$aggregations5, _result$aggregations6, _result$aggregations7, _result$aggregations8, _source$slo;
    const dateRange = (0, _date_range.toDateRange)(slo.timeWindow);
    const isDefinedWithGroupBy = ![slo.groupBy].flat().includes(_sloSchema.ALL_VALUE);
    const hasInstanceId = instanceId !== _sloSchema.ALL_VALUE;
    const shouldIncludeInstanceIdFilter = isDefinedWithGroupBy && hasInstanceId;
    const instanceIdFilter = shouldIncludeInstanceIdFilter ? [{
      term: {
        'slo.instanceId': instanceId
      }
    }] : [];
    const result = await this.esClient.search({
      index: remoteName ? `${remoteName}:${_constants.SLO_DESTINATION_INDEX_PATTERN}` : _constants.SLO_DESTINATION_INDEX_PATTERN,
      size: 0,
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': slo.id
            }
          }, {
            term: {
              'slo.revision': slo.revision
            }
          }, {
            range: {
              '@timestamp': {
                gte: dateRange.from.toISOString(),
                lt: dateRange.to.toISOString()
              }
            }
          }, ...instanceIdFilter]
        }
      },
      aggs: {
        ...(shouldIncludeInstanceIdFilter && {
          last_doc: {
            top_hits: {
              sort: [{
                '@timestamp': {
                  order: 'desc'
                }
              }],
              _source: {
                includes: ['slo.groupings', 'monitor', 'observer', 'config_id']
              },
              size: 1
            }
          }
        }),
        ...(_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod) && {
          good: {
            sum: {
              field: 'slo.isGoodSlice'
            }
          },
          total: {
            value_count: {
              field: 'slo.isGoodSlice'
            }
          }
        }),
        ...(_sloSchema.occurrencesBudgetingMethodSchema.is(slo.budgetingMethod) && {
          good: {
            sum: {
              field: 'slo.numerator'
            }
          },
          total: {
            sum: {
              field: 'slo.denominator'
            }
          }
        })
      }
    });
    const good = (_result$aggregations$ = (_result$aggregations = result.aggregations) === null || _result$aggregations === void 0 ? void 0 : (_result$aggregations$2 = _result$aggregations.good) === null || _result$aggregations$2 === void 0 ? void 0 : _result$aggregations$2.value) !== null && _result$aggregations$ !== void 0 ? _result$aggregations$ : 0;
    const total = (_result$aggregations$3 = (_result$aggregations2 = result.aggregations) === null || _result$aggregations2 === void 0 ? void 0 : (_result$aggregations3 = _result$aggregations2.total) === null || _result$aggregations3 === void 0 ? void 0 : _result$aggregations3.value) !== null && _result$aggregations$3 !== void 0 ? _result$aggregations$3 : 0;
    const source = (_result$aggregations4 = result.aggregations) === null || _result$aggregations4 === void 0 ? void 0 : (_result$aggregations5 = _result$aggregations4.last_doc) === null || _result$aggregations5 === void 0 ? void 0 : (_result$aggregations6 = _result$aggregations5.hits) === null || _result$aggregations6 === void 0 ? void 0 : (_result$aggregations7 = _result$aggregations6.hits) === null || _result$aggregations7 === void 0 ? void 0 : (_result$aggregations8 = _result$aggregations7[0]) === null || _result$aggregations8 === void 0 ? void 0 : _result$aggregations8._source;
    const groupings = source === null || source === void 0 ? void 0 : (_source$slo = source.slo) === null || _source$slo === void 0 ? void 0 : _source$slo.groupings;
    let sliValue;
    if (_sloSchema.timeslicesBudgetingMethodSchema.is(slo.budgetingMethod)) {
      const totalSlices = (0, _compute_total_slices_from_date_range.computeTotalSlicesFromDateRange)(dateRange, slo.objective.timesliceWindow);
      sliValue = (0, _services.computeSLI)(good, total, totalSlices);
    } else {
      sliValue = (0, _services.computeSLI)(good, total);
    }
    const initialErrorBudget = 1 - slo.objective.target;
    const consumedErrorBudget = sliValue < 0 ? 0 : (1 - sliValue) / initialErrorBudget;
    const errorBudget = (0, _services.toErrorBudget)(initialErrorBudget, consumedErrorBudget, _sloSchema.calendarAlignedTimeWindowSchema.is(slo.timeWindow) && _sloSchema.occurrencesBudgetingMethodSchema.is(slo.budgetingMethod));
    return {
      summary: {
        sliValue,
        errorBudget,
        status: (0, _services.computeSummaryStatus)(slo.objective, sliValue, errorBudget)
      },
      groupings: groupings ? (0, _utils.getFlattenedGroupings)({
        groupBy: slo.groupBy,
        groupings
      }) : {},
      meta: getMetaFields(slo, source !== null && source !== void 0 ? source : {})
    };
  }
}
exports.DefaultSummaryClient = DefaultSummaryClient;
function getMetaFields(slo, source) {
  var _source$monitor$id, _source$monitor, _source$observer$name, _source$observer, _source$config_id;
  const {
    indicator: {
      type
    }
  } = slo;
  switch (type) {
    case 'sli.synthetics.availability':
      return {
        synthetics: {
          monitorId: (_source$monitor$id = (_source$monitor = source.monitor) === null || _source$monitor === void 0 ? void 0 : _source$monitor.id) !== null && _source$monitor$id !== void 0 ? _source$monitor$id : '',
          locationId: (_source$observer$name = (_source$observer = source.observer) === null || _source$observer === void 0 ? void 0 : _source$observer.name) !== null && _source$observer$name !== void 0 ? _source$observer$name : '',
          configId: (_source$config_id = source.config_id) !== null && _source$config_id !== void 0 ? _source$config_id : ''
        }
      };
    default:
      return {};
  }
}