"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.DefaultResourceInstaller = void 0;
var _slo_mappings_template = require("../assets/component_templates/slo_mappings_template");
var _slo_settings_template = require("../assets/component_templates/slo_settings_template");
var _slo_summary_mappings_template = require("../assets/component_templates/slo_summary_mappings_template");
var _slo_summary_settings_template = require("../assets/component_templates/slo_summary_settings_template");
var _constants = require("../../common/constants");
var _slo_index_templates = require("../assets/index_templates/slo_index_templates");
var _slo_summary_index_templates = require("../assets/index_templates/slo_summary_index_templates");
var _slo_pipeline_template = require("../assets/ingest_templates/slo_pipeline_template");
var _retry = require("../utils/retry");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class DefaultResourceInstaller {
  constructor(esClient, logger) {
    this.esClient = esClient;
    this.logger = logger;
  }
  async ensureCommonResourcesInstalled() {
    try {
      this.logger.info('Installing SLO shared resources');
      await Promise.all([this.createOrUpdateComponentTemplate((0, _slo_mappings_template.getSLOMappingsTemplate)(_constants.SLO_COMPONENT_TEMPLATE_MAPPINGS_NAME)), this.createOrUpdateComponentTemplate((0, _slo_settings_template.getSLOSettingsTemplate)(_constants.SLO_COMPONENT_TEMPLATE_SETTINGS_NAME)), this.createOrUpdateComponentTemplate((0, _slo_summary_mappings_template.getSLOSummaryMappingsTemplate)(_constants.SLO_SUMMARY_COMPONENT_TEMPLATE_MAPPINGS_NAME)), this.createOrUpdateComponentTemplate((0, _slo_summary_settings_template.getSLOSummarySettingsTemplate)(_constants.SLO_SUMMARY_COMPONENT_TEMPLATE_SETTINGS_NAME))]);
      await this.createOrUpdateIndexTemplate((0, _slo_index_templates.getSLOIndexTemplate)(_constants.SLO_INDEX_TEMPLATE_NAME, _constants.SLO_INDEX_TEMPLATE_PATTERN, [_constants.SLO_COMPONENT_TEMPLATE_MAPPINGS_NAME, _constants.SLO_COMPONENT_TEMPLATE_SETTINGS_NAME]));
      await this.createOrUpdateIndexTemplate((0, _slo_summary_index_templates.getSLOSummaryIndexTemplate)(_constants.SLO_SUMMARY_INDEX_TEMPLATE_NAME, _constants.SLO_SUMMARY_INDEX_TEMPLATE_PATTERN, [_constants.SLO_SUMMARY_COMPONENT_TEMPLATE_MAPPINGS_NAME, _constants.SLO_SUMMARY_COMPONENT_TEMPLATE_SETTINGS_NAME]));
      await this.createIndex(_constants.SLO_DESTINATION_INDEX_NAME);
      await this.createIndex(_constants.SLO_SUMMARY_DESTINATION_INDEX_NAME);
      await this.createIndex(_constants.SLO_SUMMARY_TEMP_INDEX_NAME);
      await this.createOrUpdateIngestPipelineTemplate((0, _slo_pipeline_template.getSLOPipelineTemplate)(_constants.SLO_INGEST_PIPELINE_NAME, _constants.SLO_INGEST_PIPELINE_INDEX_NAME_PREFIX));
    } catch (err) {
      this.logger.error(`Error installing resources shared for SLO: ${err.message}`);
      throw err;
    }
  }
  async createOrUpdateComponentTemplate(template) {
    var _template$_meta;
    const currentVersion = await fetchComponentTemplateVersion(template.name, this.logger, this.esClient);
    if ((_template$_meta = template._meta) !== null && _template$_meta !== void 0 && _template$_meta.version && currentVersion === template._meta.version) {
      this.logger.info(`SLO component template found with version [${template._meta.version}]`);
    } else {
      this.logger.info(`Installing SLO component template [${template.name}]`);
      return this.execute(() => this.esClient.cluster.putComponentTemplate(template));
    }
  }
  async createOrUpdateIndexTemplate(template) {
    var _template$_meta2;
    const currentVersion = await fetchIndexTemplateVersion(template.name, this.logger, this.esClient);
    if ((_template$_meta2 = template._meta) !== null && _template$_meta2 !== void 0 && _template$_meta2.version && currentVersion === template._meta.version) {
      this.logger.info(`SLO index template found with version [${template._meta.version}]`);
    } else {
      this.logger.info(`Installing SLO index template [${template.name}]`);
      return this.execute(() => this.esClient.indices.putIndexTemplate(template));
    }
  }
  async createOrUpdateIngestPipelineTemplate(template) {
    var _template$_meta3;
    const currentVersion = await fetchIngestPipelineVersion(template.id, this.logger, this.esClient);
    if ((_template$_meta3 = template._meta) !== null && _template$_meta3 !== void 0 && _template$_meta3.version && currentVersion === template._meta.version) {
      this.logger.info(`SLO ingest pipeline found with version [${template._meta.version}]`);
    } else {
      this.logger.info(`Installing SLO ingest pipeline [${template.id}]`);
      return this.execute(() => this.esClient.ingest.putPipeline(template));
    }
  }
  async createIndex(indexName) {
    try {
      await this.execute(() => this.esClient.indices.create({
        index: indexName
      }));
    } catch (err) {
      var _err$meta, _err$meta$body, _err$meta$body$error;
      if ((err === null || err === void 0 ? void 0 : (_err$meta = err.meta) === null || _err$meta === void 0 ? void 0 : (_err$meta$body = _err$meta.body) === null || _err$meta$body === void 0 ? void 0 : (_err$meta$body$error = _err$meta$body.error) === null || _err$meta$body$error === void 0 ? void 0 : _err$meta$body$error.type) !== 'resource_already_exists_exception') {
        throw err;
      }
    }
  }
  async execute(esCall) {
    return await (0, _retry.retryTransientEsErrors)(esCall, {
      logger: this.logger
    });
  }
}
exports.DefaultResourceInstaller = DefaultResourceInstaller;
async function fetchComponentTemplateVersion(name, logger, esClient) {
  var _getTemplateRes$compo, _getTemplateRes$compo2, _getTemplateRes$compo3, _getTemplateRes$compo4;
  const getTemplateRes = await (0, _retry.retryTransientEsErrors)(() => esClient.cluster.getComponentTemplate({
    name
  }, {
    ignore: [404]
  }), {
    logger
  });
  return (getTemplateRes === null || getTemplateRes === void 0 ? void 0 : (_getTemplateRes$compo = getTemplateRes.component_templates) === null || _getTemplateRes$compo === void 0 ? void 0 : (_getTemplateRes$compo2 = _getTemplateRes$compo[0]) === null || _getTemplateRes$compo2 === void 0 ? void 0 : (_getTemplateRes$compo3 = _getTemplateRes$compo2.component_template) === null || _getTemplateRes$compo3 === void 0 ? void 0 : (_getTemplateRes$compo4 = _getTemplateRes$compo3._meta) === null || _getTemplateRes$compo4 === void 0 ? void 0 : _getTemplateRes$compo4.version) || null;
}
async function fetchIndexTemplateVersion(name, logger, esClient) {
  var _getTemplateRes$index, _getTemplateRes$index2, _getTemplateRes$index3, _getTemplateRes$index4;
  const getTemplateRes = await (0, _retry.retryTransientEsErrors)(() => esClient.indices.getIndexTemplate({
    name
  }, {
    ignore: [404]
  }), {
    logger
  });
  return (getTemplateRes === null || getTemplateRes === void 0 ? void 0 : (_getTemplateRes$index = getTemplateRes.index_templates) === null || _getTemplateRes$index === void 0 ? void 0 : (_getTemplateRes$index2 = _getTemplateRes$index[0]) === null || _getTemplateRes$index2 === void 0 ? void 0 : (_getTemplateRes$index3 = _getTemplateRes$index2.index_template) === null || _getTemplateRes$index3 === void 0 ? void 0 : (_getTemplateRes$index4 = _getTemplateRes$index3._meta) === null || _getTemplateRes$index4 === void 0 ? void 0 : _getTemplateRes$index4.version) || null;
}
async function fetchIngestPipelineVersion(id, logger, esClient) {
  var _getPipelineRes$id, _getPipelineRes$id$_m;
  const getPipelineRes = await (0, _retry.retryTransientEsErrors)(() => esClient.ingest.getPipeline({
    id
  }, {
    ignore: [404]
  }), {
    logger
  });
  return (getPipelineRes === null || getPipelineRes === void 0 ? void 0 : (_getPipelineRes$id = getPipelineRes[id]) === null || _getPipelineRes$id === void 0 ? void 0 : (_getPipelineRes$id$_m = _getPipelineRes$id._meta) === null || _getPipelineRes$id$_m === void 0 ? void 0 : _getPipelineRes$id$_m.version) || null;
}