"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ResetSLO = void 0;
var _sloSchema = require("@kbn/slo-schema");
var _constants = require("../../common/constants");
var _slo_summary_pipeline_template = require("../assets/ingest_templates/slo_summary_pipeline_template");
var _retry = require("../utils/retry");
var _create_temp_summary = require("./summary_transform_generator/helpers/create_temp_summary");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class ResetSLO {
  constructor(esClient, repository, transformManager, summaryTransformManager, logger, spaceId, basePath) {
    this.esClient = esClient;
    this.repository = repository;
    this.transformManager = transformManager;
    this.summaryTransformManager = summaryTransformManager;
    this.logger = logger;
    this.spaceId = spaceId;
    this.basePath = basePath;
  }
  async execute(sloId) {
    const slo = await this.repository.findById(sloId);
    const summaryTransformId = (0, _constants.getSLOSummaryTransformId)(slo.id, slo.revision);
    await this.summaryTransformManager.stop(summaryTransformId);
    await this.summaryTransformManager.uninstall(summaryTransformId);
    const rollupTransformId = (0, _constants.getSLOTransformId)(slo.id, slo.revision);
    await this.transformManager.stop(rollupTransformId);
    await this.transformManager.uninstall(rollupTransformId);
    await Promise.all([this.deleteRollupData(slo.id), this.deleteSummaryData(slo.id)]);
    try {
      await this.transformManager.install(slo);
      await this.transformManager.start(rollupTransformId);
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.ingest.putPipeline((0, _slo_summary_pipeline_template.getSLOSummaryPipelineTemplate)(slo, this.spaceId, this.basePath)), {
        logger: this.logger
      });
      await this.summaryTransformManager.install(slo);
      await this.summaryTransformManager.start(summaryTransformId);
      await (0, _retry.retryTransientEsErrors)(() => this.esClient.index({
        index: _constants.SLO_SUMMARY_TEMP_INDEX_NAME,
        id: `slo-${slo.id}`,
        document: (0, _create_temp_summary.createTempSummaryDocument)(slo, this.spaceId, this.basePath),
        refresh: true
      }), {
        logger: this.logger
      });
    } catch (err) {
      this.logger.error(`Cannot reset the SLO [id: ${slo.id}, revision: ${slo.revision}]. Rolling back.`);
      await this.summaryTransformManager.stop(summaryTransformId);
      await this.summaryTransformManager.uninstall(summaryTransformId);
      await this.transformManager.stop(rollupTransformId);
      await this.transformManager.uninstall(rollupTransformId);
      await this.esClient.ingest.deletePipeline({
        id: (0, _constants.getSLOSummaryPipelineId)(slo.id, slo.revision)
      }, {
        ignore: [404]
      });
      throw err;
    }
    const updatedSlo = await this.repository.save({
      ...slo,
      version: _constants.SLO_MODEL_VERSION,
      updatedAt: new Date()
    });
    return _sloSchema.resetSLOResponseSchema.encode(updatedSlo);
  }

  /**
   * Deleting all SLI rollup data matching the sloId. All revision will be deleted in case of
   * residual documents.
   *
   * @param sloId
   */
  async deleteRollupData(sloId) {
    await this.esClient.deleteByQuery({
      index: _constants.SLO_DESTINATION_INDEX_PATTERN,
      refresh: true,
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': sloId
            }
          }]
        }
      }
    });
  }

  /**
   * Deleting the summary documents matching the sloId. All revision will be deleted in case of
   * residual documents.
   *
   * @param sloId
   */
  async deleteSummaryData(sloId) {
    await this.esClient.deleteByQuery({
      index: _constants.SLO_SUMMARY_DESTINATION_INDEX_PATTERN,
      refresh: true,
      query: {
        bool: {
          filter: [{
            term: {
              'slo.id': sloId
            }
          }]
        }
      }
    });
  }
}
exports.ResetSLO = ResetSLO;