"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryTimelineTaskConfig = createTelemetryTimelineTaskConfig;
var _constants = require("../constants");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createTelemetryTimelineTaskConfig() {
  const taskName = 'Security Solution Timeline telemetry';
  const taskType = 'security:telemetry-timelines';
  return {
    type: taskType,
    title: taskName,
    interval: '1h',
    timeout: '15m',
    version: '1.0.1',
    runTask: async (taskId, logger, receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const log = (0, _helpers.newTelemetryLogger)(logger.get('timelines'));
      const fetcher = new _helpers.TelemetryTimelineFetcher(receiver);
      const trace = taskMetricsService.start(taskType);
      log.l(`Running task: ${taskId} [last: ${taskExecutionPeriod.last} - current: ${taskExecutionPeriod.current}]`);
      try {
        let counter = 0;
        const {
          rangeFrom,
          rangeTo
        } = (0, _helpers.ranges)(taskExecutionPeriod);
        const alertsIndex = receiver.getAlertsIndex();
        if (!alertsIndex) {
          throw Error('alerts index is not ready yet, skipping telemetry task');
        }
        const alerts = await receiver.fetchTimelineAlerts(alertsIndex, rangeFrom, rangeTo);
        log.l(`found ${alerts.length} alerts to process`);
        for (const alert of alerts) {
          var _sender$getTelemetryU, _sender$getTelemetryU2;
          const result = await fetcher.fetchTimeline(alert);
          (_sender$getTelemetryU = sender.getTelemetryUsageCluster()) === null || _sender$getTelemetryU === void 0 ? void 0 : _sender$getTelemetryU.incrementCounter({
            counterName: 'telemetry_timeline',
            counterType: 'timeline_node_count',
            incrementBy: result.nodes
          });
          (_sender$getTelemetryU2 = sender.getTelemetryUsageCluster()) === null || _sender$getTelemetryU2 === void 0 ? void 0 : _sender$getTelemetryU2.incrementCounter({
            counterName: 'telemetry_timeline',
            counterType: 'timeline_event_count',
            incrementBy: result.events
          });
          if (result.timeline) {
            sender.sendOnDemand(_constants.TELEMETRY_CHANNEL_TIMELINE, [result.timeline]);
            counter += 1;
          } else {
            log.l('no events in timeline');
          }
        }
        log.l(`sent ${counter} timelines. Concluding timeline task.`);
        taskMetricsService.end(trace);
        return counter;
      } catch (err) {
        taskMetricsService.end(trace, err);
        return 0;
      }
    }
  };
}