"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createTelemetryConfigurationTaskConfig = createTelemetryConfigurationTaskConfig;
var _types = require("../types");
var _artifact = require("../artifact");
var _configuration = require("../configuration");
var _helpers = require("../helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function createTelemetryConfigurationTaskConfig() {
  const taskName = 'Security Solution Telemetry Configuration Task';
  const taskType = 'security:telemetry-configuration';
  return {
    type: taskType,
    title: taskName,
    interval: '1h',
    timeout: '1m',
    version: '1.0.0',
    runTask: async (taskId, logger, _receiver, sender, taskMetricsService, taskExecutionPeriod) => {
      const log = (0, _helpers.newTelemetryLogger)(logger.get('configuration'));
      const trace = taskMetricsService.start(taskType);
      log.l(`Running task: ${taskId} [last: ${taskExecutionPeriod.last} - current: ${taskExecutionPeriod.current}]`);
      try {
        const artifactName = 'telemetry-buffer-and-batch-sizes-v1';
        const manifest = await _artifact.artifactService.getArtifact(artifactName);
        if (manifest.notModified) {
          log.l('No new configuration artifact found, skipping...');
          taskMetricsService.end(trace);
          return 0;
        }
        const configArtifact = manifest.data;
        log.l(`Got telemetry configuration artifact: ${JSON.stringify(configArtifact)}`);
        _configuration.telemetryConfiguration.max_detection_alerts_batch = configArtifact.max_detection_alerts_batch;
        _configuration.telemetryConfiguration.telemetry_max_buffer_size = configArtifact.telemetry_max_buffer_size;
        _configuration.telemetryConfiguration.max_detection_rule_telemetry_batch = configArtifact.max_detection_rule_telemetry_batch;
        _configuration.telemetryConfiguration.max_endpoint_telemetry_batch = configArtifact.max_endpoint_telemetry_batch;
        _configuration.telemetryConfiguration.max_security_list_telemetry_batch = configArtifact.max_security_list_telemetry_batch;
        if (configArtifact.use_async_sender) {
          _configuration.telemetryConfiguration.use_async_sender = configArtifact.use_async_sender;
        }
        if (configArtifact.sender_channels) {
          log.l('Updating sender channels configuration');
          _configuration.telemetryConfiguration.sender_channels = configArtifact.sender_channels;
          const channelsDict = Object.values(_types.TelemetryChannel).reduce((acc, channel) => acc.set(channel, channel), new Map());
          Object.entries(configArtifact.sender_channels).forEach(([channelName, config]) => {
            if (channelName === 'default') {
              log.l('Updating default configuration');
              sender.updateDefaultQueueConfig({
                bufferTimeSpanMillis: config.buffer_time_span_millis,
                inflightEventsThreshold: config.inflight_events_threshold,
                maxPayloadSizeBytes: config.max_payload_size_bytes
              });
            } else {
              const channel = channelsDict.get(channelName);
              if (!channel) {
                log.l(`Ignoring unknown channel "${channelName}"`);
              } else {
                log.l(`Updating configuration for channel "${channelName}`);
                sender.updateQueueConfig(channel, {
                  bufferTimeSpanMillis: config.buffer_time_span_millis,
                  inflightEventsThreshold: config.inflight_events_threshold,
                  maxPayloadSizeBytes: config.max_payload_size_bytes
                });
              }
            }
          });
        }
        if (configArtifact.pagination_config) {
          log.l('Updating pagination configuration');
          _configuration.telemetryConfiguration.pagination_config = configArtifact.pagination_config;
          _receiver.setMaxPageSizeBytes(configArtifact.pagination_config.max_page_size_bytes);
          _receiver.setNumDocsToSample(configArtifact.pagination_config.num_docs_to_sample);
        }
        taskMetricsService.end(trace);
        log.l(`Updated TelemetryConfiguration: ${JSON.stringify(_configuration.telemetryConfiguration)}`);
        return 0;
      } catch (err) {
        log.l(`Failed to set telemetry configuration due to ${err.message}`);
        _configuration.telemetryConfiguration.resetAllToDefault();
        taskMetricsService.end(trace, err);
        return 0;
      }
    }
  };
}