"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.TelemetryReceiver = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _os = _interopRequireDefault(require("os"));
var _lodash = require("lodash");
var _securitysolutionListConstants = require("@kbn/securitysolution-list-constants");
var _securitysolutionRules = require("@kbn/securitysolution-rules");
var _moment = _interopRequireDefault(require("moment"));
var _helpers = require("./helpers");
var _fetch = require("../../endpoint/routes/resolver/tree/utils/fetch");
var _configuration = require("./configuration");
var _constants = require("../../../common/constants");
var _constants2 = require("../../../common/detection_engine/constants");
var _constants3 = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class TelemetryReceiver {
  constructor(logger) {
    (0, _defineProperty2.default)(this, "logger", void 0);
    (0, _defineProperty2.default)(this, "agentClient", void 0);
    (0, _defineProperty2.default)(this, "agentPolicyService", void 0);
    (0, _defineProperty2.default)(this, "_esClient", void 0);
    (0, _defineProperty2.default)(this, "exceptionListClient", void 0);
    (0, _defineProperty2.default)(this, "soClient", void 0);
    (0, _defineProperty2.default)(this, "getIndexForType", void 0);
    (0, _defineProperty2.default)(this, "alertsIndex", void 0);
    (0, _defineProperty2.default)(this, "clusterInfo", void 0);
    (0, _defineProperty2.default)(this, "processTreeFetcher", void 0);
    (0, _defineProperty2.default)(this, "packageService", void 0);
    (0, _defineProperty2.default)(this, "experimentalFeatures", void 0);
    (0, _defineProperty2.default)(this, "maxRecords", 10_000);
    // default to 2% of host's total memory or 80MiB, whichever is smaller
    (0, _defineProperty2.default)(this, "maxPageSizeBytes", Math.min(_os.default.totalmem() * 0.02, 80 * 1024 * 1024));
    // number of docs to query to estimate the size of a single doc
    (0, _defineProperty2.default)(this, "numDocsToSample", 10);
    this.logger = (0, _helpers.newTelemetryLogger)(logger.get('telemetry_events.receiver'));
  }
  async start(core, getIndexForType, alertsIndex, endpointContextService, exceptionListClient, packageService) {
    this.getIndexForType = getIndexForType;
    this.alertsIndex = alertsIndex;
    this.agentClient = endpointContextService === null || endpointContextService === void 0 ? void 0 : endpointContextService.getInternalFleetServices().agent;
    this.agentPolicyService = endpointContextService === null || endpointContextService === void 0 ? void 0 : endpointContextService.getInternalFleetServices().agentPolicy;
    this._esClient = core === null || core === void 0 ? void 0 : core.elasticsearch.client.asInternalUser;
    this.exceptionListClient = exceptionListClient;
    this.packageService = packageService;
    this.soClient = core === null || core === void 0 ? void 0 : core.savedObjects.createInternalRepository();
    this.clusterInfo = await this.fetchClusterInfo();
    this.experimentalFeatures = endpointContextService === null || endpointContextService === void 0 ? void 0 : endpointContextService.experimentalFeatures;
    const elasticsearch = core === null || core === void 0 ? void 0 : core.elasticsearch.client;
    this.processTreeFetcher = new _fetch.Fetcher(elasticsearch);
    (0, _helpers.setClusterInfo)(this.clusterInfo);
  }
  getClusterInfo() {
    return this.clusterInfo;
  }
  getAlertsIndex() {
    return this.alertsIndex;
  }
  getExperimentalFeatures() {
    return this.experimentalFeatures;
  }
  async fetchDetectionRulesPackageVersion() {
    var _this$packageService;
    return (_this$packageService = this.packageService) === null || _this$packageService === void 0 ? void 0 : _this$packageService.asInternalUser.getInstallation(_constants2.PREBUILT_RULES_PACKAGE_NAME);
  }
  async fetchFleetAgents() {
    var _this$agentClient$lis, _this$agentClient;
    if (this.esClient === undefined || this.esClient === null) {
      throw Error('elasticsearch client is unavailable: cannot retrieve fleet agents');
    }
    return (_this$agentClient$lis = (_this$agentClient = this.agentClient) === null || _this$agentClient === void 0 ? void 0 : _this$agentClient.listAgents({
      perPage: this.maxRecords,
      showInactive: true,
      sortField: 'enrolled_at',
      sortOrder: 'desc'
    }).then(response => {
      var _response$agents;
      const agents = (_response$agents = response === null || response === void 0 ? void 0 : response.agents) !== null && _response$agents !== void 0 ? _response$agents : [];
      return agents.reduce((cache, agent) => {
        if (agent.policy_id !== null && agent.policy_id !== undefined) {
          cache.set(agent.id, agent.policy_id);
        }
        return cache;
      }, new Map());
    })) !== null && _this$agentClient$lis !== void 0 ? _this$agentClient$lis : new Map();
  }
  async fetchEndpointPolicyResponses(executeFrom, executeTo) {
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: `.ds-metrics-endpoint.policy*`,
      ignore_unavailable: false,
      body: {
        size: 0,
        // no query results required - only aggregation quantity
        query: {
          range: {
            '@timestamp': {
              gte: executeFrom,
              lt: executeTo
            }
          }
        },
        aggs: {
          policy_responses: {
            terms: {
              size: this.maxRecords,
              field: 'agent.id'
            },
            aggs: {
              latest_response: {
                top_hits: {
                  size: 1,
                  _source: {
                    includes: ['agent', 'event', 'Endpoint.policy.applied.status', 'Endpoint.policy.applied.actions', 'Endpoint.policy.applied.artifacts.global', 'Endpoint.configuration', 'Endpoint.state']
                  },
                  sort: [{
                    '@timestamp': {
                      order: 'desc'
                    }
                  }]
                }
              }
            }
          }
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    }).then(response => response.body).then(failedPolicyResponses => {
      var _failedPolicyResponse, _failedPolicyResponse2, _failedPolicyResponse3;
      const buckets = (_failedPolicyResponse = failedPolicyResponses === null || failedPolicyResponses === void 0 ? void 0 : (_failedPolicyResponse2 = failedPolicyResponses.aggregations) === null || _failedPolicyResponse2 === void 0 ? void 0 : (_failedPolicyResponse3 = _failedPolicyResponse2.policy_responses) === null || _failedPolicyResponse3 === void 0 ? void 0 : _failedPolicyResponse3.buckets) !== null && _failedPolicyResponse !== void 0 ? _failedPolicyResponse : [];

      // If there is no policy responses in the 24h > now then we will continue
      return buckets.reduce((cache, endpointAgentId) => cache.set(endpointAgentId.key, endpointAgentId.latest_response.hits.hits[0]._source), new Map());
    });
  }
  async fetchEndpointMetricsAbstract(executeFrom, executeTo) {
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: _constants.ENDPOINT_METRICS_INDEX,
      ignore_unavailable: false,
      body: {
        size: 0,
        // no query results required - only aggregation quantity
        query: {
          range: {
            '@timestamp': {
              gte: executeFrom,
              lt: executeTo
            }
          }
        },
        aggs: {
          endpoint_agents: {
            terms: {
              field: 'agent.id',
              size: this.maxRecords
            },
            aggs: {
              latest_metrics: {
                top_hits: {
                  size: 1,
                  _source: {
                    excludes: ['*']
                  },
                  sort: [{
                    '@timestamp': {
                      order: 'desc'
                    }
                  }]
                }
              }
            }
          },
          endpoint_count: {
            cardinality: {
              field: 'agent.id'
            }
          }
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    }).then(response => response.body).then(endpointMetricsResponse => {
      var _endpointMetricsRespo, _endpointMetricsRespo2, _endpointMetricsRespo3;
      const buckets = (_endpointMetricsRespo = endpointMetricsResponse === null || endpointMetricsResponse === void 0 ? void 0 : (_endpointMetricsRespo2 = endpointMetricsResponse.aggregations) === null || _endpointMetricsRespo2 === void 0 ? void 0 : (_endpointMetricsRespo3 = _endpointMetricsRespo2.endpoint_agents) === null || _endpointMetricsRespo3 === void 0 ? void 0 : _endpointMetricsRespo3.buckets) !== null && _endpointMetricsRespo !== void 0 ? _endpointMetricsRespo : [];
      const endpointMetricIds = buckets.map(epMetrics => epMetrics.latest_metrics.hits.hits[0]._id);
      const totalEndpoints = buckets.length;
      return {
        endpointMetricIds,
        totalEndpoints
      };
    });
  }
  fetchEndpointMetricsById(ids) {
    const query = {
      sort: [{
        '@timestamp': {
          order: 'desc'
        }
      }],
      query: {
        ids: {
          values: ids
        }
      },
      _source: {
        includes: ['@timestamp', 'agent', 'Endpoint.metrics', 'elastic.agent', 'host', 'event']
      }
    };
    return this.paginate(_constants.ENDPOINT_METRICS_INDEX, query);
  }
  async fetchEndpointMetadata(executeFrom, executeTo) {
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: `.ds-metrics-endpoint.metadata-*`,
      ignore_unavailable: false,
      body: {
        size: 0,
        // no query results required - only aggregation quantity
        query: {
          range: {
            '@timestamp': {
              gte: executeFrom,
              lt: executeTo
            }
          }
        },
        aggs: {
          endpoint_metadata: {
            terms: {
              field: 'agent.id',
              size: this.maxRecords
            },
            aggs: {
              latest_metadata: {
                top_hits: {
                  size: 1,
                  _source: {
                    includes: ['@timestamp', 'agent', 'Endpoint.capabilities', 'elastic.agent']
                  },
                  sort: [{
                    '@timestamp': {
                      order: 'desc'
                    }
                  }]
                }
              }
            }
          }
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    }).then(response => response.body).then(endpointMetadataResponse => {
      var _endpointMetadataResp, _endpointMetadataResp2, _endpointMetadataResp3;
      const buckets = (_endpointMetadataResp = endpointMetadataResponse === null || endpointMetadataResponse === void 0 ? void 0 : (_endpointMetadataResp2 = endpointMetadataResponse.aggregations) === null || _endpointMetadataResp2 === void 0 ? void 0 : (_endpointMetadataResp3 = _endpointMetadataResp2.endpoint_metadata) === null || _endpointMetadataResp3 === void 0 ? void 0 : _endpointMetadataResp3.buckets) !== null && _endpointMetadataResp !== void 0 ? _endpointMetadataResp : [];
      return buckets.reduce((cache, endpointAgentId) => {
        const doc = endpointAgentId.latest_metadata.hits.hits[0]._source;
        cache.set(endpointAgentId.key, doc);
        return cache;
      }, new Map());
    });
  }
  async *fetchDiagnosticAlertsBatch(executeFrom, executeTo) {
    this.logger.l('Searching diagnostic alerts', {
      from: executeFrom,
      to: executeTo
    });
    let pitId = await this.openPointInTime(_constants3.DEFAULT_DIAGNOSTIC_INDEX);
    let fetchMore = true;
    let searchAfter;
    const query = {
      query: {
        range: {
          'event.ingested': {
            gte: executeFrom,
            lt: executeTo
          }
        }
      },
      track_total_hits: false,
      sort: [{
        'event.ingested': {
          order: 'desc'
        }
      }],
      pit: {
        id: pitId
      },
      search_after: searchAfter,
      size: _configuration.telemetryConfiguration.telemetry_max_buffer_size
    };
    let response = null;
    while (fetchMore) {
      var _response3, _response4;
      try {
        var _response;
        response = await this.esClient().search(query);
        const numOfHits = (_response = response) === null || _response === void 0 ? void 0 : _response.hits.hits.length;
        if (numOfHits > 0) {
          var _response2;
          const lastHit = (_response2 = response) === null || _response2 === void 0 ? void 0 : _response2.hits.hits[numOfHits - 1];
          query.search_after = lastHit === null || lastHit === void 0 ? void 0 : lastHit.sort;
        } else {
          fetchMore = false;
        }
        this.logger.l('Diagnostic alerts to return', {
          numOfHits
        });
        fetchMore = numOfHits > 0 && numOfHits < _configuration.telemetryConfiguration.telemetry_max_buffer_size;
      } catch (e) {
        this.logger.l('Error fetching alerts', {
          error: JSON.stringify(e)
        });
        fetchMore = false;
      }
      if (response == null) {
        await this.closePointInTime(pitId);
        return;
      }
      const alerts = (_response3 = response) === null || _response3 === void 0 ? void 0 : _response3.hits.hits.flatMap(h => h._source != null ? [h._source] : []);
      if (((_response4 = response) === null || _response4 === void 0 ? void 0 : _response4.pit_id) != null) {
        var _response5;
        pitId = (_response5 = response) === null || _response5 === void 0 ? void 0 : _response5.pit_id;
      }
      yield alerts;
    }
    this.closePointInTime(pitId);
  }
  async fetchPolicyConfigs(id) {
    var _this$agentPolicyServ;
    if (this.soClient === undefined || this.soClient === null) {
      throw Error('saved object client is unavailable: cannot retrieve endpoint policy configurations');
    }
    return (_this$agentPolicyServ = this.agentPolicyService) === null || _this$agentPolicyServ === void 0 ? void 0 : _this$agentPolicyServ.get(this.soClient, id);
  }
  async fetchTrustedApplications() {
    var _results$total, _results$page, _results$per_page;
    if ((this === null || this === void 0 ? void 0 : this.exceptionListClient) === undefined || (this === null || this === void 0 ? void 0 : this.exceptionListClient) === null) {
      throw Error('exception list client is unavailable: cannot retrieve trusted applications');
    }

    // Ensure list is created if it does not exist
    await this.exceptionListClient.createTrustedAppsList();
    const timeFrom = _moment.default.utc().subtract(1, 'day').valueOf();
    const results = await this.exceptionListClient.findExceptionListItem({
      listId: _securitysolutionListConstants.ENDPOINT_ARTIFACT_LISTS.trustedApps.id,
      page: 1,
      perPage: 10_000,
      filter: `exception-list-agnostic.attributes.created_at >= ${timeFrom}`,
      namespaceType: 'agnostic',
      sortField: 'name',
      sortOrder: 'asc'
    });
    return {
      data: results === null || results === void 0 ? void 0 : results.data.map(_helpers.trustedApplicationToTelemetryEntry),
      total: (_results$total = results === null || results === void 0 ? void 0 : results.total) !== null && _results$total !== void 0 ? _results$total : 0,
      page: (_results$page = results === null || results === void 0 ? void 0 : results.page) !== null && _results$page !== void 0 ? _results$page : 1,
      per_page: (_results$per_page = results === null || results === void 0 ? void 0 : results.per_page) !== null && _results$per_page !== void 0 ? _results$per_page : this.maxRecords
    };
  }
  async fetchEndpointList(listId) {
    var _results$data$map, _results$total2, _results$page2, _results$per_page2;
    if ((this === null || this === void 0 ? void 0 : this.exceptionListClient) === undefined || (this === null || this === void 0 ? void 0 : this.exceptionListClient) === null) {
      throw Error('exception list client is unavailable: could not retrieve trusted applications');
    }

    // Ensure list is created if it does not exist
    await this.exceptionListClient.createEndpointList();
    const timeFrom = _moment.default.utc().subtract(1, 'day').valueOf();
    const results = await this.exceptionListClient.findExceptionListItem({
      listId,
      page: 1,
      perPage: this.maxRecords,
      filter: `exception-list-agnostic.attributes.created_at >= ${timeFrom}`,
      namespaceType: 'agnostic',
      sortField: 'name',
      sortOrder: 'asc'
    });
    return {
      data: (_results$data$map = results === null || results === void 0 ? void 0 : results.data.map(_helpers.exceptionListItemToTelemetryEntry)) !== null && _results$data$map !== void 0 ? _results$data$map : [],
      total: (_results$total2 = results === null || results === void 0 ? void 0 : results.total) !== null && _results$total2 !== void 0 ? _results$total2 : 0,
      page: (_results$page2 = results === null || results === void 0 ? void 0 : results.page) !== null && _results$page2 !== void 0 ? _results$page2 : 1,
      per_page: (_results$per_page2 = results === null || results === void 0 ? void 0 : results.per_page) !== null && _results$per_page2 !== void 0 ? _results$per_page2 : this.maxRecords
    };
  }

  /**
   * Gets the elastic rules which are the rules that have immutable set to true and are of a particular rule type
   * @returns The elastic rules
   */
  async fetchDetectionRules() {
    var _this$getIndexForType;
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: (_this$getIndexForType = this.getIndexForType) === null || _this$getIndexForType === void 0 ? void 0 : _this$getIndexForType.call(this, 'alert'),
      ignore_unavailable: true,
      body: {
        size: this.maxRecords,
        query: {
          bool: {
            must: [{
              bool: {
                filter: {
                  terms: {
                    'alert.alertTypeId': [_securitysolutionRules.SIGNALS_ID, _securitysolutionRules.EQL_RULE_TYPE_ID, _securitysolutionRules.ESQL_RULE_TYPE_ID, _securitysolutionRules.ML_RULE_TYPE_ID, _securitysolutionRules.QUERY_RULE_TYPE_ID, _securitysolutionRules.SAVED_QUERY_RULE_TYPE_ID, _securitysolutionRules.INDICATOR_RULE_TYPE_ID, _securitysolutionRules.THRESHOLD_RULE_TYPE_ID, _securitysolutionRules.NEW_TERMS_RULE_TYPE_ID]
                  }
                }
              }
            }, {
              bool: {
                filter: {
                  terms: {
                    'alert.params.immutable': [true]
                  }
                }
              }
            }]
          }
        }
      }
    };
    return this.esClient().search(query, {
      meta: true
    });
  }
  async fetchDetectionExceptionList(listId, ruleVersion) {
    var _this$exceptionListCl, _results$data$map2, _results$total3, _results$page3, _results$per_page3;
    if ((this === null || this === void 0 ? void 0 : this.exceptionListClient) === undefined || (this === null || this === void 0 ? void 0 : this.exceptionListClient) === null) {
      throw Error('exception list client is unavailable: could not retrieve trusted applications');
    }

    // Ensure list is created if it does not exist
    await this.exceptionListClient.createTrustedAppsList();
    const timeFrom = `exception-list.attributes.created_at >= ${_moment.default.utc().subtract(24, 'hours').valueOf()}`;
    const results = await ((_this$exceptionListCl = this.exceptionListClient) === null || _this$exceptionListCl === void 0 ? void 0 : _this$exceptionListCl.findExceptionListsItem({
      listId: [listId],
      filter: [timeFrom],
      perPage: this.maxRecords,
      page: 1,
      sortField: 'exception-list.created_at',
      sortOrder: 'desc',
      namespaceType: ['single']
    }));
    return {
      data: (_results$data$map2 = results === null || results === void 0 ? void 0 : results.data.map(r => (0, _helpers.ruleExceptionListItemToTelemetryEvent)(r, ruleVersion))) !== null && _results$data$map2 !== void 0 ? _results$data$map2 : [],
      total: (_results$total3 = results === null || results === void 0 ? void 0 : results.total) !== null && _results$total3 !== void 0 ? _results$total3 : 0,
      page: (_results$page3 = results === null || results === void 0 ? void 0 : results.page) !== null && _results$page3 !== void 0 ? _results$page3 : 1,
      per_page: (_results$per_page3 = results === null || results === void 0 ? void 0 : results.per_page) !== null && _results$per_page3 !== void 0 ? _results$per_page3 : this.maxRecords
    };
  }
  async *fetchPrebuiltRuleAlertsBatch(executeFrom, executeTo) {
    this.logger.l('Searching prebuilt rule alerts from', {
      executeFrom,
      executeTo
    });
    let pitId = await this.openPointInTime(_constants3.DEFAULT_DIAGNOSTIC_INDEX);
    let fetchMore = true;
    let searchAfter;
    const query = {
      query: {
        bool: {
          filter: [{
            bool: {
              should: [{
                bool: {
                  must_not: {
                    bool: {
                      should: [{
                        match_phrase: {
                          'kibana.alert.rule.name': 'Malware Prevention Alert'
                        }
                      }]
                    }
                  }
                }
              }, {
                bool: {
                  must_not: {
                    bool: {
                      should: [{
                        match_phrase: {
                          'kibana.alert.rule.name': 'Malware Detection Alert'
                        }
                      }]
                    }
                  }
                }
              }, {
                bool: {
                  must_not: {
                    bool: {
                      should: [{
                        match_phrase: {
                          'kibana.alert.rule.name': 'Ransomware Prevention Alert'
                        }
                      }]
                    }
                  }
                }
              }, {
                bool: {
                  must_not: {
                    bool: {
                      should: [{
                        match_phrase: {
                          'kibana.alert.rule.name': 'Ransomware Detection Alert'
                        }
                      }]
                    }
                  }
                }
              }]
            }
          }, {
            bool: {
              should: [{
                match_phrase: {
                  'kibana.alert.rule.parameters.immutable': 'true'
                }
              }]
            }
          }, {
            range: {
              '@timestamp': {
                gte: executeFrom,
                lte: executeTo
              }
            }
          }]
        }
      },
      track_total_hits: false,
      sort: [{
        '@timestamp': {
          order: 'asc',
          format: 'strict_date_optional_time_nanos'
        }
      }, {
        _shard_doc: 'desc'
      }],
      pit: {
        id: pitId
      },
      search_after: searchAfter,
      size: 1_000
    };
    let response = null;
    try {
      while (fetchMore) {
        var _response6, _response8;
        response = await this.esClient().search(query);
        const numOfHits = (_response6 = response) === null || _response6 === void 0 ? void 0 : _response6.hits.hits.length;
        if (numOfHits > 0) {
          var _response7;
          const lastHit = (_response7 = response) === null || _response7 === void 0 ? void 0 : _response7.hits.hits[numOfHits - 1];
          query.search_after = lastHit === null || lastHit === void 0 ? void 0 : lastHit.sort;
        } else {
          fetchMore = false;
        }
        fetchMore = numOfHits > 0 && numOfHits < 1_000;
        if (response == null) {
          await this.closePointInTime(pitId);
          return;
        }
        const alerts = response.hits.hits.flatMap(h => h._source != null ? [h._source] : []);
        if (((_response8 = response) === null || _response8 === void 0 ? void 0 : _response8.pit_id) != null) {
          var _response9;
          pitId = (_response9 = response) === null || _response9 === void 0 ? void 0 : _response9.pit_id;
        }
        this.logger.l('Prebuilt rule alerts to return', {
          alerts: alerts.length
        });
        yield alerts;
      }
    } catch (e) {
      // to keep backward compatibility with the previous implementation, silent return
      // once we start using `paginate` this error should be managed downstream
      this.logger.l('Error fetching alerts', {
        error: JSON.stringify(e)
      });
      return;
    } finally {
      await this.closePointInTime(pitId);
    }
  }
  async openPointInTime(indexPattern) {
    const keepAlive = '5m';
    const pitId = (await this.esClient().openPointInTime({
      index: `${indexPattern}*`,
      keep_alive: keepAlive,
      expand_wildcards: ['open', 'hidden']
    })).id;
    return pitId;
  }
  async closePointInTime(pitId) {
    try {
      await this.esClient().closePointInTime({
        id: pitId
      });
    } catch (error) {
      this.logger.l('Error trying to close point in time', {
        pit: pitId,
        error: JSON.stringify(error)
      });
    }
  }
  async fetchTimelineAlerts(index, rangeFrom, rangeTo) {
    // default is from looking at Kibana saved objects and online documentation
    const keepAlive = '5m';

    // create and assign an initial point in time
    let pitId = (await this.esClient().openPointInTime({
      index: `${index}*`,
      keep_alive: keepAlive
    })).id;
    let fetchMore = true;
    let searchAfter;
    let alertsToReturn = [];
    while (fetchMore) {
      var _response12, _response13;
      const query = {
        query: {
          bool: {
            filter: [{
              bool: {
                should: [{
                  match_phrase: {
                    'event.module': 'endpoint'
                  }
                }]
              }
            }, {
              bool: {
                should: [{
                  match_phrase: {
                    'kibana.alert.rule.parameters.immutable': 'true'
                  }
                }]
              }
            }, {
              range: {
                '@timestamp': {
                  gte: rangeFrom,
                  lte: rangeTo
                }
              }
            }]
          }
        },
        aggs: {
          endpoint_alert_count: {
            cardinality: {
              field: 'event.id'
            }
          }
        },
        track_total_hits: false,
        sort: [{
          '@timestamp': {
            order: 'asc',
            format: 'strict_date_optional_time_nanos'
          }
        }, {
          _shard_doc: 'desc'
        }],
        pit: {
          id: pitId
        },
        search_after: searchAfter,
        size: 1000
      };
      let response = null;
      try {
        var _response10;
        response = await this.esClient().search(query);
        const numOfHits = (_response10 = response) === null || _response10 === void 0 ? void 0 : _response10.hits.hits.length;
        if (numOfHits > 0) {
          var _response11;
          const lastHit = (_response11 = response) === null || _response11 === void 0 ? void 0 : _response11.hits.hits[numOfHits - 1];
          searchAfter = lastHit === null || lastHit === void 0 ? void 0 : lastHit.sort;
        }
        fetchMore = numOfHits > 0;
      } catch (e) {
        this.logger.l('Error fetching alerts', {
          error: JSON.stringify(e)
        });
        fetchMore = false;
      }
      const alerts = (_response12 = response) === null || _response12 === void 0 ? void 0 : _response12.hits.hits;
      alertsToReturn = alertsToReturn.concat(alerts !== null && alerts !== void 0 ? alerts : []);
      if (((_response13 = response) === null || _response13 === void 0 ? void 0 : _response13.pit_id) != null) {
        var _response14;
        pitId = (_response14 = response) === null || _response14 === void 0 ? void 0 : _response14.pit_id;
      }
    }
    try {
      await this.esClient().closePointInTime({
        id: pitId
      });
    } catch (error) {
      this.logger.l('Error trying to close point in time', {
        pit: pitId,
        error: JSON.stringify(error),
        keepAlive
      });
    }
    this.logger.l('Timeline alerts to return', {
      alerts: alertsToReturn.length
    });
    return alertsToReturn || [];
  }
  async buildProcessTree(entityId, resolverSchema, startOfDay, endOfDay) {
    if (this.processTreeFetcher === undefined || this.processTreeFetcher === null) {
      throw Error('resolver tree builder is unavailable: cannot build encoded endpoint event graph');
    }
    const request = {
      ancestors: 200,
      descendants: 500,
      timeRange: {
        from: startOfDay,
        to: endOfDay
      },
      schema: resolverSchema,
      nodes: [entityId],
      indexPatterns: [`${this.alertsIndex}*`, 'logs-*'],
      descendantLevels: 20
    };
    return this.processTreeFetcher.tree(request, true);
  }
  async fetchTimelineEvents(nodeIds) {
    const query = {
      expand_wildcards: ['open', 'hidden'],
      index: [`${this.alertsIndex}*`, 'logs-*'],
      ignore_unavailable: true,
      body: {
        size: 100,
        _source: {
          include: ['@timestamp', 'process', 'event', 'file', 'network', 'dns', 'kibana.rule.alert.uuid']
        },
        query: {
          bool: {
            filter: [{
              terms: {
                'process.entity_id': nodeIds
              }
            }, {
              term: {
                'event.category': 'process'
              }
            }]
          }
        }
      }
    };
    return this.esClient().search(query);
  }
  async fetchValueListMetaData(_interval) {
    var _this$getIndexForType2, _this$getIndexForType3;
    const listQuery = {
      expand_wildcards: ['open', 'hidden'],
      index: '.lists-*',
      ignore_unavailable: true,
      body: {
        size: 0,
        // no query results required - only aggregation quantity
        aggs: {
          total_value_list_count: {
            cardinality: {
              field: 'name'
            }
          },
          type_breakdown: {
            terms: {
              field: 'type',
              size: 50
            }
          }
        }
      }
    };
    const itemQuery = {
      expand_wildcards: ['open', 'hidden'],
      index: '.items-*',
      ignore_unavailable: true,
      body: {
        size: 0,
        // no query results required - only aggregation quantity
        aggs: {
          value_list_item_count: {
            terms: {
              field: 'list_id',
              size: 100
            }
          }
        }
      }
    };
    const exceptionListQuery = {
      expand_wildcards: ['open', 'hidden'],
      index: (_this$getIndexForType2 = this.getIndexForType) === null || _this$getIndexForType2 === void 0 ? void 0 : _this$getIndexForType2.call(this, 'exception-list'),
      ignore_unavailable: true,
      body: {
        size: 0,
        // no query results required - only aggregation quantity
        query: {
          bool: {
            must: [{
              match: {
                'exception-list.entries.type': 'list'
              }
            }]
          }
        },
        aggs: {
          vl_included_in_exception_lists_count: {
            cardinality: {
              field: 'exception-list.entries.list.id'
            }
          }
        }
      }
    };
    const indicatorMatchRuleQuery = {
      expand_wildcards: ['open', 'hidden'],
      index: (_this$getIndexForType3 = this.getIndexForType) === null || _this$getIndexForType3 === void 0 ? void 0 : _this$getIndexForType3.call(this, 'alert'),
      ignore_unavailable: true,
      body: {
        size: 0,
        query: {
          bool: {
            must: [{
              prefix: {
                'alert.params.threatIndex': '.items'
              }
            }]
          }
        },
        aggs: {
          vl_used_in_indicator_match_rule_count: {
            cardinality: {
              field: 'alert.params.ruleId'
            }
          }
        }
      }
    };
    const [listMetrics, itemMetrics, exceptionListMetrics, indicatorMatchMetrics] = await Promise.all([this.esClient().search(listQuery), this.esClient().search(itemQuery), this.esClient().search(exceptionListQuery), this.esClient().search(indicatorMatchRuleQuery)]);
    const listMetricsResponse = listMetrics;
    const itemMetricsResponse = itemMetrics;
    const exceptionListMetricsResponse = exceptionListMetrics;
    const indicatorMatchMetricsResponse = indicatorMatchMetrics;
    return {
      listMetricsResponse,
      itemMetricsResponse,
      exceptionListMetricsResponse,
      indicatorMatchMetricsResponse
    };
  }
  async fetchClusterInfo() {
    // @ts-expect-error version.build_date is of type estypes.DateTime
    return this.esClient().info();
  }
  async fetchLicenseInfo() {
    try {
      const ret = await this.esClient().transport.request({
        method: 'GET',
        path: '/_license',
        querystring: {
          local: true
        }
      });
      return ret.license;
    } catch (err) {
      this.logger.l('failed retrieving license', {
        error: JSON.stringify(err)
      });
      return undefined;
    }
  }

  // calculates the number of documents that can be returned per page
  // or "-1" if the query returns no documents
  async docsPerPage(index, query) {
    const sampleQuery = {
      query: (0, _lodash.cloneDeep)(query.query),
      size: this.numDocsToSample,
      index
    };
    const sampleSizeBytes = await this.esClient().search(sampleQuery).then(r => r.hits.hits.reduce((sum, hit) => JSON.stringify(hit._source).length + sum, 0));
    const docSizeBytes = sampleSizeBytes / this.numDocsToSample;
    if (docSizeBytes === 0) {
      return -1;
    }
    return Math.max(Math.floor(this.maxPageSizeBytes / docSizeBytes), 1);
  }
  async *paginate(index, query) {
    if (query.sort == null) {
      throw Error('Not possible to paginate a query without a sort attribute');
    }
    const size = await this.docsPerPage(index, query);
    if (size === -1) {
      return;
    }
    const pit = {
      id: await this.openPointInTime(index)
    };
    const esQuery = {
      ...(0, _lodash.cloneDeep)(query),
      pit,
      size: Math.min(size, 10_000)
    };
    try {
      do {
        var _response$hits$hits$l, _response$hits$hits;
        const response = await this.nextPage(esQuery);
        const hits = (_response$hits$hits$l = response === null || response === void 0 ? void 0 : response.hits.hits.length) !== null && _response$hits$hits$l !== void 0 ? _response$hits$hits$l : 0;
        if (hits === 0) {
          return;
        }
        esQuery.search_after = response === null || response === void 0 ? void 0 : (_response$hits$hits = response.hits.hits[hits - 1]) === null || _response$hits$hits === void 0 ? void 0 : _response$hits$hits.sort;
        const data = response === null || response === void 0 ? void 0 : response.hits.hits.flatMap(h => h._source != null ? [h._source] : []);
        yield data;
      } while (esQuery.search_after !== undefined);
    } catch (e) {
      this.logger.l('Error running paginated query', {
        error: JSON.stringify(e)
      });
      throw e;
    } finally {
      await this.closePointInTime(pit.id);
    }
  }
  async nextPage(esQuery) {
    return this.esClient().search(esQuery);
  }
  setMaxPageSizeBytes(bytes) {
    this.maxPageSizeBytes = bytes;
  }
  setNumDocsToSample(n) {
    this.numDocsToSample = n;
  }
  esClient() {
    if (this._esClient === undefined || this._esClient === null) {
      throw Error('elasticsearch client is unavailable');
    }
    return this._esClient;
  }
}
exports.TelemetryReceiver = TelemetryReceiver;