"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.riskScoreCalculationRoute = void 0;
var _utils = require("@kbn/lists-plugin/server/routes/utils");
var _securitysolutionEsUtils = require("@kbn/securitysolution-es-utils");
var _constants = require("../../../../../common/constants");
var _request_schema = require("../../../../../common/entity_analytics/risk_engine/risk_score_calculation/request_schema");
var _route_validation = require("../../../../utils/build_validation/route_validation");
var _asset_criticality = require("../../asset_criticality");
var _risk_score_service = require("../risk_score_service");
var _get_risk_inputs_index = require("../get_risk_inputs_index");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const riskScoreCalculationRoute = (router, logger) => {
  router.versioned.post({
    path: _constants.RISK_SCORE_CALCULATION_URL,
    access: 'internal',
    options: {
      tags: ['access:securitySolution', `access:${_constants.APP_ID}-entity-analytics`]
    }
  }).addVersion({
    version: '1',
    validate: {
      request: {
        body: (0, _route_validation.buildRouteValidation)(_request_schema.riskScoreCalculationRequestSchema)
      }
    }
  }, async (context, request, response) => {
    const siemResponse = (0, _utils.buildSiemResponse)(response);
    const securityContext = await context.securitySolution;
    const coreContext = await context.core;
    const esClient = coreContext.elasticsearch.client.asCurrentUser;
    const soClient = coreContext.savedObjects.client;
    const spaceId = securityContext.getSpaceId();
    const riskEngineDataClient = securityContext.getRiskEngineDataClient();
    const riskScoreDataClient = securityContext.getRiskScoreDataClient();
    const assetCriticalityDataClient = securityContext.getAssetCriticalityDataClient();
    const securityConfig = await securityContext.getConfig();
    const assetCriticalityService = (0, _asset_criticality.assetCriticalityServiceFactory)({
      assetCriticalityDataClient,
      uiSettingsClient: coreContext.uiSettings.client
    });
    const riskScoreService = (0, _risk_score_service.riskScoreServiceFactory)({
      assetCriticalityService,
      esClient,
      logger,
      riskEngineDataClient,
      riskScoreDataClient,
      spaceId
    });
    const {
      after_keys: userAfterKeys,
      data_view_id: dataViewId,
      debug,
      page_size: userPageSize,
      identifier_type: identifierType,
      filter,
      range,
      weights
    } = request.body;
    try {
      const {
        index,
        runtimeMappings
      } = await (0, _get_risk_inputs_index.getRiskInputsIndex)({
        dataViewId,
        logger,
        soClient
      });
      const afterKeys = userAfterKeys !== null && userAfterKeys !== void 0 ? userAfterKeys : {};
      const pageSize = userPageSize !== null && userPageSize !== void 0 ? userPageSize : _constants.DEFAULT_RISK_SCORE_PAGE_SIZE;
      const entityAnalyticsConfig = await riskScoreService.getConfigurationWithDefaults(securityConfig.entityAnalytics);
      const alertSampleSizePerShard = entityAnalyticsConfig === null || entityAnalyticsConfig === void 0 ? void 0 : entityAnalyticsConfig.alertSampleSizePerShard;
      const result = await riskScoreService.calculateAndPersistScores({
        afterKeys,
        debug,
        pageSize,
        identifierType,
        index,
        filter,
        range,
        runtimeMappings,
        weights,
        alertSampleSizePerShard
      });
      return response.ok({
        body: result
      });
    } catch (e) {
      const error = (0, _securitysolutionEsUtils.transformError)(e);
      return siemResponse.error({
        statusCode: error.statusCode,
        body: {
          message: error.message,
          full_error: JSON.stringify(e)
        },
        bypassErrorFormat: true
      });
    }
  });
};
exports.riskScoreCalculationRoute = riskScoreCalculationRoute;