"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.calculateRiskScores = void 0;
var _lodash = require("lodash");
var _technical_rule_data_field_names = require("@kbn/rule-registry-plugin/common/technical_rule_data_field_names");
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _with_security_span = require("../../../utils/with_security_span");
var _helpers = require("../asset_criticality/helpers");
var _helpers2 = require("./helpers");
var _risk_weights = require("./risk_weights");
var _constants = require("./constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const formatForResponse = ({
  bucket,
  criticality,
  now,
  identifierField,
  includeNewFields
}) => {
  const riskDetails = bucket.top_inputs.risk_details;
  const criticalityModifier = (0, _helpers.getCriticalityModifier)(criticality === null || criticality === void 0 ? void 0 : criticality.criticality_level);
  const normalizedScoreWithCriticality = (0, _helpers.applyCriticalityToScore)({
    score: riskDetails.value.normalized_score,
    modifier: criticalityModifier
  });
  const calculatedLevel = (0, _risk_engine.getRiskLevel)(normalizedScoreWithCriticality);
  const categoryTwoScore = normalizedScoreWithCriticality - riskDetails.value.normalized_score;
  const categoryTwoCount = criticalityModifier ? 1 : 0;
  const newFields = {
    category_2_score: categoryTwoScore,
    category_2_count: categoryTwoCount,
    criticality_level: criticality === null || criticality === void 0 ? void 0 : criticality.criticality_level,
    criticality_modifier: criticalityModifier
  };
  return {
    '@timestamp': now,
    id_field: identifierField,
    id_value: bucket.key[identifierField],
    calculated_level: calculatedLevel,
    calculated_score: riskDetails.value.score,
    calculated_score_norm: normalizedScoreWithCriticality,
    category_1_score: (0, _helpers.normalize)({
      number: riskDetails.value.category_1_score,
      max: _constants.RISK_SCORING_SUM_MAX
    }),
    category_1_count: riskDetails.value.category_1_count,
    notes: riskDetails.value.notes,
    inputs: riskDetails.value.risk_inputs.map(riskInput => {
      var _riskInput$rule_name;
      return {
        id: riskInput.id,
        index: riskInput.index,
        description: `Alert from Rule: ${(_riskInput$rule_name = riskInput.rule_name) !== null && _riskInput$rule_name !== void 0 ? _riskInput$rule_name : 'RULE_NOT_FOUND'}`,
        category: _risk_engine.RiskCategories.category_1,
        risk_score: riskInput.score,
        timestamp: riskInput.time,
        contribution_score: riskInput.contribution
      };
    }),
    ...(includeNewFields ? newFields : {})
  };
};
const filterFromRange = range => ({
  range: {
    '@timestamp': {
      lt: range.end,
      gte: range.start
    }
  }
});
const buildReduceScript = ({
  globalIdentifierTypeWeight
}) => {
  return `
    Map results = new HashMap();
    List inputs = [];
    for (state in states) {
      inputs.addAll(state.inputs)
    }
    Collections.sort(inputs, (a, b) -> b.get('weighted_score').compareTo(a.get('weighted_score')));

    double num_inputs_to_score = Math.min(inputs.length, params.max_risk_inputs_per_identity);
    results['notes'] = [];
    if (num_inputs_to_score == params.max_risk_inputs_per_identity) {
      results['notes'].add('Number of risk inputs (' + inputs.length + ') exceeded the maximum allowed (' + params.max_risk_inputs_per_identity + ').');
    }

    ${(0, _risk_weights.buildCategoryScoreDeclarations)()}
    ${(0, _risk_weights.buildCategoryCountDeclarations)()}

    double total_score = 0;
    double current_score = 0;
    List risk_inputs = [];
    for (int i = 0; i < num_inputs_to_score; i++) {
      current_score = inputs[i].weighted_score / Math.pow(i + 1, params.p);

      if (i < ${_constants.MAX_INPUTS_COUNT}) {
        inputs[i]["contribution"] = 100 * current_score / params.risk_cap;
        risk_inputs.add(inputs[i]);
      }

      ${(0, _risk_weights.buildCategoryAssignment)()}
      total_score += current_score;
    }

    ${globalIdentifierTypeWeight != null ? `total_score *= ${globalIdentifierTypeWeight};` : ''}
    double score_norm = 100 * total_score / params.risk_cap;
    results['score'] = total_score;
    results['normalized_score'] = score_norm;
    results['risk_inputs'] = risk_inputs;

    return results;
  `;
};
const buildIdentifierTypeAggregation = ({
  afterKeys,
  identifierType,
  pageSize,
  weights,
  alertSampleSizePerShard
}) => {
  const globalIdentifierTypeWeight = (0, _risk_weights.getGlobalWeightForIdentifierType)({
    identifierType,
    weights
  });
  const identifierField = (0, _helpers2.getFieldForIdentifierAgg)(identifierType);
  return {
    composite: {
      size: pageSize,
      sources: [{
        [identifierField]: {
          terms: {
            field: identifierField
          }
        }
      }],
      after: (0, _helpers2.getAfterKeyForIdentifierType)({
        identifierType,
        afterKeys
      })
    },
    aggs: {
      top_inputs: {
        sampler: {
          shard_size: alertSampleSizePerShard
        },
        aggs: {
          risk_details: {
            scripted_metric: {
              init_script: 'state.inputs = []',
              map_script: `
                Map fields = new HashMap();
                String category = doc['${_technical_rule_data_field_names.EVENT_KIND}'].value;
                double score = doc['${_technical_rule_data_field_names.ALERT_RISK_SCORE}'].value;
                double weighted_score = 0.0;
          
                fields.put('time', doc['@timestamp'].value);
                fields.put('rule_name', doc['${_technical_rule_data_field_names.ALERT_RULE_NAME}'].value);

                fields.put('category', category);
                fields.put('index', doc['_index'].value);
                fields.put('id', doc['${_technical_rule_data_field_names.ALERT_UUID}'].value);
                fields.put('score', score);
                
                ${(0, _risk_weights.buildWeightingOfScoreByCategory)({
                userWeights: weights,
                identifierType
              })}
                fields.put('weighted_score', weighted_score);
          
                state.inputs.add(fields);
              `,
              combine_script: 'return state;',
              params: {
                max_risk_inputs_per_identity: _constants.RISK_SCORING_INPUTS_COUNT_MAX,
                p: _constants.RISK_SCORING_SUM_VALUE,
                risk_cap: _constants.RISK_SCORING_SUM_MAX
              },
              reduce_script: buildReduceScript({
                globalIdentifierTypeWeight
              })
            }
          }
        }
      }
    }
  };
};
const processScores = async ({
  assetCriticalityService,
  buckets,
  identifierField,
  logger,
  now
}) => {
  if (buckets.length === 0) {
    return [];
  }
  const isAssetCriticalityEnabled = await assetCriticalityService.isEnabled();
  if (!isAssetCriticalityEnabled) {
    return buckets.map(bucket => formatForResponse({
      bucket,
      now,
      identifierField,
      includeNewFields: false
    }));
  }
  const identifiers = buckets.map(bucket => ({
    id_field: identifierField,
    id_value: bucket.key[identifierField]
  }));
  let criticalities = [];
  try {
    criticalities = await assetCriticalityService.getCriticalitiesByIdentifiers(identifiers);
  } catch (e) {
    logger.warn(`Error retrieving criticality: ${e}. Scoring will proceed without criticality information.`);
  }
  return buckets.map(bucket => {
    const criticality = criticalities.find(c => c.id_field === identifierField && c.id_value === bucket.key[identifierField]);
    return formatForResponse({
      bucket,
      criticality,
      identifierField,
      now,
      includeNewFields: true
    });
  });
};
const calculateRiskScores = async ({
  afterKeys: userAfterKeys,
  assetCriticalityService,
  debug,
  esClient,
  filter: userFilter,
  identifierType,
  index,
  logger,
  pageSize,
  range,
  runtimeMappings,
  weights,
  alertSampleSizePerShard = 10_000
}) => (0, _with_security_span.withSecuritySpan)('calculateRiskScores', async () => {
  var _response$aggregation, _response$aggregation2, _response$aggregation3, _response$aggregation4, _response$aggregation5, _response$aggregation6;
  const now = new Date().toISOString();
  const filter = [filterFromRange(range), {
    bool: {
      must_not: {
        term: {
          [_technical_rule_data_field_names.ALERT_WORKFLOW_STATUS]: 'closed'
        }
      }
    }
  }, {
    exists: {
      field: _technical_rule_data_field_names.ALERT_RISK_SCORE
    }
  }];
  if (!(0, _lodash.isEmpty)(userFilter)) {
    filter.push(userFilter);
  }
  const identifierTypes = identifierType ? [identifierType] : ['host', 'user'];
  const request = {
    size: 0,
    _source: false,
    index,
    runtime_mappings: runtimeMappings,
    query: {
      function_score: {
        query: {
          bool: {
            filter,
            should: [{
              match_all: {} // This forces ES to calculate score
            }]
          }
        },
        field_value_factor: {
          field: _technical_rule_data_field_names.ALERT_RISK_SCORE // sort by risk score
        }
      }
    },
    aggs: identifierTypes.reduce((aggs, _identifierType) => {
      aggs[_identifierType] = buildIdentifierTypeAggregation({
        afterKeys: userAfterKeys,
        identifierType: _identifierType,
        pageSize,
        weights,
        alertSampleSizePerShard
      });
      return aggs;
    }, {})
  };
  if (debug) {
    logger.info(`Executing Risk Score query:\n${JSON.stringify(request)}`);
  }
  const response = await esClient.search(request);
  if (debug) {
    logger.info(`Received Risk Score response:\n${JSON.stringify(response)}`);
  }
  if (response.aggregations == null) {
    return {
      ...(debug ? {
        request,
        response
      } : {}),
      after_keys: {},
      scores: {
        host: [],
        user: []
      }
    };
  }
  const userBuckets = (_response$aggregation = (_response$aggregation2 = response.aggregations.user) === null || _response$aggregation2 === void 0 ? void 0 : _response$aggregation2.buckets) !== null && _response$aggregation !== void 0 ? _response$aggregation : [];
  const hostBuckets = (_response$aggregation3 = (_response$aggregation4 = response.aggregations.host) === null || _response$aggregation4 === void 0 ? void 0 : _response$aggregation4.buckets) !== null && _response$aggregation3 !== void 0 ? _response$aggregation3 : [];
  const afterKeys = {
    host: (_response$aggregation5 = response.aggregations.host) === null || _response$aggregation5 === void 0 ? void 0 : _response$aggregation5.after_key,
    user: (_response$aggregation6 = response.aggregations.user) === null || _response$aggregation6 === void 0 ? void 0 : _response$aggregation6.after_key
  };
  const hostScores = await processScores({
    assetCriticalityService,
    buckets: hostBuckets,
    identifierField: 'host.name',
    logger,
    now
  });
  const userScores = await processScores({
    assetCriticalityService,
    buckets: userBuckets,
    identifierField: 'user.name',
    logger,
    now
  });
  return {
    ...(debug ? {
      request,
      response
    } : {}),
    after_keys: afterKeys,
    scores: {
      host: hostScores,
      user: userScores
    }
  };
});
exports.calculateRiskScores = calculateRiskScores;