"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.RiskEngineDataClient = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _risk_engine = require("../../../../common/entity_analytics/risk_engine");
var _transforms = require("../utils/transforms");
var _saved_object_configuration = require("./utils/saved_object_configuration");
var _bulk_delete_saved_objects = require("../../risk_score/prebuilt_saved_objects/helpers/bulk_delete_saved_objects");
var _tasks = require("../risk_score/tasks");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

class RiskEngineDataClient {
  constructor(options) {
    (0, _defineProperty2.default)(this, "getConfiguration", () => (0, _saved_object_configuration.getConfiguration)({
      savedObjectsClient: this.options.soClient
    }));
    this.options = options;
  }
  async init({
    namespace,
    taskManager,
    riskScoreDataClient
  }) {
    const result = {
      legacyRiskEngineDisabled: false,
      riskEngineResourcesInstalled: false,
      riskEngineConfigurationCreated: false,
      riskEngineEnabled: false,
      errors: []
    };
    try {
      result.legacyRiskEngineDisabled = await this.disableLegacyRiskEngine({
        namespace
      });
    } catch (e) {
      result.legacyRiskEngineDisabled = false;
      result.errors.push(e.message);
    }
    try {
      await riskScoreDataClient.init();
      result.riskEngineResourcesInstalled = true;
    } catch (e) {
      result.errors.push(e.message);
      return result;
    }
    try {
      await (0, _saved_object_configuration.initSavedObjects)({
        savedObjectsClient: this.options.soClient,
        namespace
      });
      result.riskEngineConfigurationCreated = true;
    } catch (e) {
      result.errors.push(e.message);
      return result;
    }

    // should be the last step, after all resources are installed
    try {
      await this.enableRiskEngine({
        taskManager
      });
      result.riskEngineEnabled = true;
    } catch (e) {
      result.errors.push(e.message);
      return result;
    }
    return result;
  }
  async getStatus({
    namespace
  }) {
    const riskEngineStatus = await this.getCurrentStatus();
    const legacyRiskEngineStatus = await this.getLegacyStatus({
      namespace
    });
    const isMaxAmountOfRiskEnginesReached = await this.getIsMaxAmountOfRiskEnginesReached();
    return {
      riskEngineStatus,
      legacyRiskEngineStatus,
      isMaxAmountOfRiskEnginesReached
    };
  }
  async enableRiskEngine({
    taskManager
  }) {
    try {
      const configurationResult = await (0, _saved_object_configuration.updateSavedObjectAttribute)({
        savedObjectsClient: this.options.soClient,
        attributes: {
          enabled: true
        }
      });
      await (0, _tasks.startRiskScoringTask)({
        logger: this.options.logger,
        namespace: this.options.namespace,
        riskEngineDataClient: this,
        taskManager
      });
      return configurationResult;
    } catch (e) {
      this.options.logger.error(`Error while enabling risk engine: ${e.message}`);
      await this.disableRiskEngine({
        taskManager
      });
      throw e;
    }
  }
  async disableRiskEngine({
    taskManager
  }) {
    await (0, _tasks.removeRiskScoringTask)({
      namespace: this.options.namespace,
      taskManager,
      logger: this.options.logger
    });
    return (0, _saved_object_configuration.updateSavedObjectAttribute)({
      savedObjectsClient: this.options.soClient,
      attributes: {
        enabled: false
      }
    });
  }
  async disableLegacyRiskEngine({
    namespace
  }) {
    const legacyRiskEngineStatus = await this.getLegacyStatus({
      namespace
    });
    if (legacyRiskEngineStatus === _risk_engine.RiskEngineStatus.NOT_INSTALLED) {
      return true;
    }
    await (0, _transforms.removeLegacyTransforms)({
      esClient: this.options.esClient,
      namespace
    });
    const deleteDashboardsPromises = [_risk_engine.RiskScoreEntity.host, _risk_engine.RiskScoreEntity.user].map(entity => (0, _bulk_delete_saved_objects.bulkDeleteSavedObjects)({
      deleteAll: true,
      savedObjectsClient: this.options.soClient,
      spaceId: namespace,
      savedObjectTemplate: `${entity}RiskScoreDashboards`
    }));
    await Promise.all(deleteDashboardsPromises);
    const newlegacyRiskEngineStatus = await this.getLegacyStatus({
      namespace
    });
    return newlegacyRiskEngineStatus === _risk_engine.RiskEngineStatus.NOT_INSTALLED;
  }
  async getCurrentStatus() {
    const configuration = await this.getConfiguration();
    if (configuration) {
      return configuration.enabled ? _risk_engine.RiskEngineStatus.ENABLED : _risk_engine.RiskEngineStatus.DISABLED;
    }
    return _risk_engine.RiskEngineStatus.NOT_INSTALLED;
  }
  async getIsMaxAmountOfRiskEnginesReached() {
    try {
      const amountOfEnabledConfigurations = await (0, _saved_object_configuration.getEnabledRiskEngineAmount)({
        savedObjectsClient: this.options.soClient
      });
      return amountOfEnabledConfigurations >= _risk_engine.MAX_SPACES_COUNT;
    } catch (e) {
      this.options.logger.error(`Error while getting amount of enabled risk engines: ${e.message}`);
      return false;
    }
  }
  async getLegacyStatus({
    namespace
  }) {
    const transforms = await (0, _transforms.getLegacyTransforms)({
      namespace,
      esClient: this.options.esClient
    });
    if (transforms.length === 0) {
      return _risk_engine.RiskEngineStatus.NOT_INSTALLED;
    }
    return _risk_engine.RiskEngineStatus.ENABLED;
  }
}
exports.RiskEngineDataClient = RiskEngineDataClient;