"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.executeBulkCreateAlerts = exports.bulkCreateSuppressedAlertsInMemory = void 0;
var _constants = require("../constants");
var _utils = require("./utils");
var _constants2 = require("../../../../../common/detection_engine/constants");
var _partition_missing_fields_events = require("./partition_missing_fields_events");
var _rule_schema = require("../../../../../common/api/detection_engine/model/rule_schema");
var _enrichments = require("./enrichments");
var _bulk_create_with_suppression = require("./bulk_create_with_suppression");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * wraps, bulk create and suppress alerts in memory, also takes care of missing fields logic.
 * If parameter alertSuppression.missingFieldsStrategy configured not to be suppressed, regular alerts will be created for such events without suppression
 */
const bulkCreateSuppressedAlertsInMemory = async ({
  enrichedEvents,
  toReturn,
  wrapHits,
  bulkCreate,
  services,
  buildReasonMessage,
  ruleExecutionLogger,
  tuple,
  alertSuppression,
  wrapSuppressedHits,
  alertWithSuppression,
  alertTimestampOverride,
  experimentalFeatures
}) => {
  var _alertSuppression$mis;
  const suppressOnMissingFields = ((_alertSuppression$mis = alertSuppression === null || alertSuppression === void 0 ? void 0 : alertSuppression.missingFieldsStrategy) !== null && _alertSuppression$mis !== void 0 ? _alertSuppression$mis : _constants2.DEFAULT_SUPPRESSION_MISSING_FIELDS_STRATEGY) === _rule_schema.AlertSuppressionMissingFieldsStrategyEnum.suppress;
  let suppressibleEvents = enrichedEvents;
  let unsuppressibleWrappedDocs = [];
  if (!suppressOnMissingFields) {
    const partitionedEvents = (0, _partition_missing_fields_events.partitionMissingFieldsEvents)(enrichedEvents, (alertSuppression === null || alertSuppression === void 0 ? void 0 : alertSuppression.groupBy) || []);
    unsuppressibleWrappedDocs = wrapHits(partitionedEvents[1], buildReasonMessage);
    suppressibleEvents = partitionedEvents[0];
  }
  const suppressibleWrappedDocs = wrapSuppressedHits(suppressibleEvents, buildReasonMessage);
  return executeBulkCreateAlerts({
    suppressibleWrappedDocs,
    unsuppressibleWrappedDocs,
    toReturn,
    bulkCreate,
    services,
    ruleExecutionLogger,
    tuple,
    alertSuppression,
    alertWithSuppression,
    alertTimestampOverride,
    experimentalFeatures
  });
};
exports.bulkCreateSuppressedAlertsInMemory = bulkCreateSuppressedAlertsInMemory;
/**
 * creates alerts in ES, both suppressed and unsuppressed
 */
const executeBulkCreateAlerts = async ({
  unsuppressibleWrappedDocs,
  suppressibleWrappedDocs,
  toReturn,
  bulkCreate,
  services,
  ruleExecutionLogger,
  tuple,
  alertSuppression,
  alertWithSuppression,
  alertTimestampOverride,
  experimentalFeatures
}) => {
  var _toReturn$suppressedA;
  // max signals for suppression includes suppressed and created alerts
  // this allows to lift max signals limitation to higher value
  // and can detects events beyond default max_signals value
  const suppressionMaxSignals = _constants.MAX_SIGNALS_SUPPRESSION_MULTIPLIER * tuple.maxSignals;
  const suppressionDuration = alertSuppression === null || alertSuppression === void 0 ? void 0 : alertSuppression.duration;
  const suppressionWindow = suppressionDuration ? `now-${suppressionDuration.value}${suppressionDuration.unit}` : tuple.from.toISOString();
  if (unsuppressibleWrappedDocs.length) {
    const unsuppressedResult = await bulkCreate(unsuppressibleWrappedDocs, tuple.maxSignals - toReturn.createdSignalsCount, (0, _enrichments.createEnrichEventsFunction)({
      services,
      logger: ruleExecutionLogger
    }));
    (0, _utils.addToSearchAfterReturn)({
      current: toReturn,
      next: unsuppressedResult
    });
  }
  const bulkCreateResult = await (0, _bulk_create_with_suppression.bulkCreateWithSuppression)({
    alertWithSuppression,
    ruleExecutionLogger,
    wrappedDocs: suppressibleWrappedDocs,
    services,
    suppressionWindow,
    alertTimestampOverride,
    isSuppressionPerRuleExecution: !suppressionDuration,
    maxAlerts: tuple.maxSignals - toReturn.createdSignalsCount,
    experimentalFeatures
  });
  (0, _utils.addToSearchAfterReturn)({
    current: toReturn,
    next: bulkCreateResult
  });
  const alertsWereTruncated = ((_toReturn$suppressedA = toReturn.suppressedAlertsCount) !== null && _toReturn$suppressedA !== void 0 ? _toReturn$suppressedA : 0) + toReturn.createdSignalsCount >= suppressionMaxSignals || toReturn.createdSignalsCount >= tuple.maxSignals;
  return {
    ...bulkCreateResult,
    alertsWereTruncated
  };
};
exports.executeBulkCreateAlerts = executeBulkCreateAlerts;