"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.createThreatSignals = void 0;
var _rxjs = require("rxjs");
var _fp = require("lodash/fp");
var _get_threat_list = require("./get_threat_list");
var _create_threat_signal = require("./create_threat_signal");
var _create_event_signal = require("./create_event_signal");
var _constants = require("../../constants");
var _utils = require("./utils");
var _get_allowed_fields_for_terms_query = require("./get_allowed_fields_for_terms_query");
var _get_event_count = require("./get_event_count");
var _get_mapping_filters = require("./get_mapping_filters");
var _constants2 = require("../../../../../../common/cti/constants");
var _utils2 = require("../../utils/utils");
var _get_fields_for_wildcard = require("../../utils/get_fields_for_wildcard");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const createThreatSignals = async ({
  alertId,
  bulkCreate,
  completeRule,
  concurrentSearches,
  eventsTelemetry,
  filters,
  inputIndex,
  itemsPerSearch,
  language,
  listClient,
  outputIndex,
  query,
  ruleExecutionLogger,
  savedId,
  searchAfterSize,
  services,
  threatFilters,
  threatIndex,
  threatIndicatorPath,
  threatLanguage,
  threatMapping,
  threatQuery,
  tuple,
  type,
  wrapHits,
  wrapSuppressedHits,
  runOpts,
  runtimeMappings,
  primaryTimestamp,
  secondaryTimestamp,
  exceptionFilter,
  unprocessedExceptions,
  inputIndexFields,
  licensing,
  experimentalFeatures
}) => {
  var _completeRule$rulePar, _completeRule$rulePar2;
  const threatMatchedFields = (0, _utils.getMatchedFields)(threatMapping);
  const allowedFieldsForTermsQuery = await (0, _get_allowed_fields_for_terms_query.getAllowedFieldsForTermQuery)({
    services,
    threatMatchedFields,
    inputIndex,
    threatIndex,
    ruleExecutionLogger
  });
  const params = completeRule.ruleParams;
  ruleExecutionLogger.debug('Indicator matching rule starting');
  const perPage = concurrentSearches * itemsPerSearch;
  const verifyExecutionCanProceed = (0, _utils.buildExecutionIntervalValidator)(completeRule.ruleConfig.schedule.interval);
  let results = {
    success: true,
    warning: false,
    enrichmentTimes: [],
    bulkCreateTimes: [],
    searchAfterTimes: [],
    lastLookBackDate: null,
    createdSignalsCount: 0,
    suppressedAlertsCount: 0,
    createdSignals: [],
    errors: [],
    warningMessages: []
  };
  const {
    eventMappingFilter,
    indicatorMappingFilter
  } = (0, _get_mapping_filters.getMappingFilters)(threatMapping);
  const allEventFilters = [...filters, eventMappingFilter];
  const allThreatFilters = [...threatFilters, indicatorMappingFilter];
  const eventCount = await (0, _get_event_count.getEventCount)({
    esClient: services.scopedClusterClient.asCurrentUser,
    index: inputIndex,
    tuple,
    query,
    language,
    filters: allEventFilters,
    primaryTimestamp,
    secondaryTimestamp,
    exceptionFilter,
    indexFields: inputIndexFields
  });
  ruleExecutionLogger.debug(`Total event count: ${eventCount}`);
  let threatPitId = (await services.scopedClusterClient.asCurrentUser.openPointInTime({
    index: threatIndex,
    keep_alive: _constants2.THREAT_PIT_KEEP_ALIVE
  })).id;
  const reassignThreatPitId = newPitId => {
    if (newPitId) threatPitId = newPitId;
  };
  const threatIndexFields = await (0, _get_fields_for_wildcard.getFieldsForWildcard)({
    index: threatIndex,
    language: threatLanguage !== null && threatLanguage !== void 0 ? threatLanguage : 'kuery',
    dataViews: services.dataViews,
    ruleExecutionLogger
  });
  const threatListCount = await (0, _get_threat_list.getThreatListCount)({
    esClient: services.scopedClusterClient.asCurrentUser,
    threatFilters: allThreatFilters,
    query: threatQuery,
    language: threatLanguage,
    index: threatIndex,
    exceptionFilter,
    indexFields: threatIndexFields
  });
  ruleExecutionLogger.debug(`Total indicator items: ${threatListCount}`);
  const threatListConfig = {
    fields: threatMapping.map(mapping => mapping.entries.map(item => item.value)).flat(),
    _source: false
  };
  const eventListConfig = {
    fields: threatMapping.map(mapping => mapping.entries.map(item => item.field)).flat(),
    _source: false
  };
  const createSignals = async ({
    getDocumentList,
    createSignal,
    totalDocumentCount
  }) => {
    let list = await getDocumentList({
      searchAfter: undefined
    });
    let documentCount = totalDocumentCount;
    while (list.hits.hits.length !== 0) {
      verifyExecutionCanProceed();
      const chunks = (0, _fp.chunk)(itemsPerSearch, list.hits.hits);
      ruleExecutionLogger.debug(`${chunks.length} concurrent indicator searches are starting.`);
      const concurrentSearchesPerformed = chunks.map(createSignal);
      const searchesPerformed = await Promise.all(concurrentSearchesPerformed);
      results = (0, _utils.combineConcurrentResults)(results, searchesPerformed);
      documentCount -= list.hits.hits.length;
      ruleExecutionLogger.debug(`Concurrent indicator match searches completed with ${results.createdSignalsCount} signals found`, `search times of ${results.searchAfterTimes}ms,`, `bulk create times ${results.bulkCreateTimes}ms,`, `all successes are ${results.success}`);

      // if alerts suppressed it means suppression enabled, so suppression alert limit should be applied (5 * max_signals)
      if (results.createdSignalsCount >= params.maxSignals) {
        if (results.warningMessages.includes((0, _utils2.getMaxSignalsWarning)())) {
          results.warningMessages = (0, _fp.uniq)(results.warningMessages);
        } else if (documentCount > 0) {
          results.warningMessages.push((0, _utils2.getMaxSignalsWarning)());
        }
        ruleExecutionLogger.debug(`Indicator match has reached its max signals count ${params.maxSignals}. Additional documents not checked are ${documentCount}`);
        break;
      } else if (results.suppressedAlertsCount && results.suppressedAlertsCount > 0 && results.suppressedAlertsCount + results.createdSignalsCount >= _constants.MAX_SIGNALS_SUPPRESSION_MULTIPLIER * params.maxSignals) {
        // warning should be already set
        ruleExecutionLogger.debug(`Indicator match has reached its max signals count ${_constants.MAX_SIGNALS_SUPPRESSION_MULTIPLIER * params.maxSignals}. Additional documents not checked are ${documentCount}`);
        break;
      }
      ruleExecutionLogger.debug(`Documents items left to check are ${documentCount}`);
      const sortIds = (0, _utils2.getSafeSortIds)(list.hits.hits[list.hits.hits.length - 1].sort);

      // ES can return negative sort id for date field, when sort order set to desc
      // this could happen when event has empty sort field
      // https://github.com/elastic/kibana/issues/174573 (happens to IM rule only since it uses desc order for events search)
      // when negative sort id used in subsequent request it fails, so when negative sort value found we don't do next request
      const hasNegativeDateSort = sortIds === null || sortIds === void 0 ? void 0 : sortIds.some(val => val < 0);
      if (hasNegativeDateSort) {
        ruleExecutionLogger.debug(`Negative date sort id value encountered: ${sortIds}. Threat search stopped.`);
        break;
      }
      list = await getDocumentList({
        searchAfter: sortIds
      });
    }
  };
  const license = await (0, _rxjs.firstValueFrom)(licensing.license$);
  const hasPlatinumLicense = license.hasAtLeast('platinum');
  const isAlertSuppressionConfigured = Boolean((_completeRule$rulePar = completeRule.ruleParams.alertSuppression) === null || _completeRule$rulePar === void 0 ? void 0 : (_completeRule$rulePar2 = _completeRule$rulePar.groupBy) === null || _completeRule$rulePar2 === void 0 ? void 0 : _completeRule$rulePar2.length);
  const isAlertSuppressionActive = isAlertSuppressionConfigured && hasPlatinumLicense;

  // alert suppression needs to be performed on results searched in ascending order, so alert's suppression boundaries would be set correctly
  // at the same time, there are concerns on performance of IM rule when sorting is set to asc, as it may lead to longer rule runs, since it will
  // first go through alerts that might ve been processed in earlier executions, when look back interval set to large values (it can't be larger than 24h)
  const sortOrder = isAlertSuppressionConfigured ? 'asc' : 'desc';
  if (eventCount < threatListCount) {
    await createSignals({
      totalDocumentCount: eventCount,
      getDocumentList: async ({
        searchAfter
      }) => (0, _get_event_count.getEventList)({
        services,
        ruleExecutionLogger,
        filters: allEventFilters,
        query,
        language,
        index: inputIndex,
        searchAfter,
        perPage,
        tuple,
        runtimeMappings,
        primaryTimestamp,
        secondaryTimestamp,
        exceptionFilter,
        eventListConfig,
        indexFields: inputIndexFields,
        sortOrder
      }),
      createSignal: slicedChunk => (0, _create_event_signal.createEventSignal)({
        alertId,
        bulkCreate,
        completeRule,
        currentEventList: slicedChunk,
        currentResult: results,
        eventsTelemetry,
        filters: allEventFilters,
        inputIndex,
        language,
        listClient,
        outputIndex,
        query,
        reassignThreatPitId,
        ruleExecutionLogger,
        savedId,
        searchAfterSize,
        services,
        threatFilters: allThreatFilters,
        threatIndex,
        threatIndicatorPath,
        threatLanguage,
        threatMapping,
        threatPitId,
        threatQuery,
        tuple,
        type,
        wrapHits,
        wrapSuppressedHits,
        runtimeMappings,
        primaryTimestamp,
        secondaryTimestamp,
        exceptionFilter,
        unprocessedExceptions,
        allowedFieldsForTermsQuery,
        threatMatchedFields,
        inputIndexFields,
        threatIndexFields,
        runOpts,
        sortOrder,
        isAlertSuppressionActive,
        experimentalFeatures
      })
    });
  } else {
    await createSignals({
      totalDocumentCount: threatListCount,
      getDocumentList: async ({
        searchAfter
      }) => (0, _get_threat_list.getThreatList)({
        esClient: services.scopedClusterClient.asCurrentUser,
        threatFilters: allThreatFilters,
        query: threatQuery,
        language: threatLanguage,
        index: threatIndex,
        searchAfter,
        ruleExecutionLogger,
        perPage,
        threatListConfig,
        pitId: threatPitId,
        reassignPitId: reassignThreatPitId,
        runtimeMappings,
        listClient,
        exceptionFilter,
        indexFields: threatIndexFields
      }),
      createSignal: slicedChunk => (0, _create_threat_signal.createThreatSignal)({
        alertId,
        bulkCreate,
        completeRule,
        currentResult: results,
        currentThreatList: slicedChunk,
        eventsTelemetry,
        filters: allEventFilters,
        inputIndex,
        language,
        listClient,
        outputIndex,
        query,
        ruleExecutionLogger,
        savedId,
        searchAfterSize,
        services,
        threatMapping,
        tuple,
        type,
        wrapHits,
        wrapSuppressedHits,
        runtimeMappings,
        primaryTimestamp,
        secondaryTimestamp,
        exceptionFilter,
        unprocessedExceptions,
        threatFilters: allThreatFilters,
        threatIndex,
        threatIndicatorPath,
        threatLanguage,
        threatPitId,
        threatQuery,
        reassignThreatPitId,
        allowedFieldsForTermsQuery,
        inputIndexFields,
        threatIndexFields,
        runOpts,
        sortOrder,
        isAlertSuppressionActive,
        experimentalFeatures
      })
    });
  }
  try {
    await services.scopedClusterClient.asCurrentUser.closePointInTime({
      id: threatPitId
    });
  } catch (error) {
    // Don't fail due to a bad point in time closure. We have seen failures in e2e tests during nominal operations.
    ruleExecutionLogger.warn(`Error trying to close point in time: "${threatPitId}", it will expire within "${_constants2.THREAT_PIT_KEEP_ALIVE}". Error is: "${error}"`);
  }
  ruleExecutionLogger.debug('Indicator matching rule has completed');
  return results;
};
exports.createThreatSignals = createThreatSignals;