"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.eqlExecutor = void 0;
var _perf_hooks = require("perf_hooks");
var _build_eql_search_request = require("./build_eql_search_request");
var _enrichments = require("../utils/enrichments");
var _utils = require("../utils/utils");
var _reason_formatters = require("../utils/reason_formatters");
var _with_security_span = require("../../../../utils/with_security_span");
var _bulk_create_suppressed_alerts_in_memory = require("../utils/bulk_create_suppressed_alerts_in_memory");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const eqlExecutor = async ({
  inputIndex,
  runtimeMappings,
  completeRule,
  tuple,
  ruleExecutionLogger,
  services,
  version,
  bulkCreate,
  wrapHits,
  wrapSequences,
  primaryTimestamp,
  secondaryTimestamp,
  exceptionFilter,
  unprocessedExceptions,
  wrapSuppressedHits,
  alertTimestampOverride,
  alertWithSuppression,
  isAlertSuppressionActive,
  experimentalFeatures
}) => {
  const ruleParams = completeRule.ruleParams;
  return (0, _with_security_span.withSecuritySpan)('eqlExecutor', async () => {
    const result = (0, _utils.createSearchAfterReturnType)();
    const request = (0, _build_eql_search_request.buildEqlSearchRequest)({
      query: ruleParams.query,
      index: inputIndex,
      from: tuple.from.toISOString(),
      to: tuple.to.toISOString(),
      size: ruleParams.maxSignals,
      filters: ruleParams.filters,
      primaryTimestamp,
      secondaryTimestamp,
      runtimeMappings,
      eventCategoryOverride: ruleParams.eventCategoryOverride,
      timestampField: ruleParams.timestampField,
      tiebreakerField: ruleParams.tiebreakerField,
      exceptionFilter
    });
    ruleExecutionLogger.debug(`EQL query request: ${JSON.stringify(request)}`);
    const exceptionsWarning = (0, _utils.getUnprocessedExceptionsWarnings)(unprocessedExceptions);
    if (exceptionsWarning) {
      result.warningMessages.push(exceptionsWarning);
    }
    const eqlSignalSearchStart = _perf_hooks.performance.now();
    try {
      var _newSignals;
      const response = await services.scopedClusterClient.asCurrentUser.eql.search(request);
      const eqlSignalSearchEnd = _perf_hooks.performance.now();
      const eqlSearchDuration = (0, _utils.makeFloatString)(eqlSignalSearchEnd - eqlSignalSearchStart);
      result.searchAfterTimes = [eqlSearchDuration];
      let newSignals;
      const {
        events,
        sequences
      } = response.hits;
      if (events) {
        if (isAlertSuppressionActive) {
          await (0, _bulk_create_suppressed_alerts_in_memory.bulkCreateSuppressedAlertsInMemory)({
            enrichedEvents: events,
            toReturn: result,
            wrapHits,
            bulkCreate,
            services,
            buildReasonMessage: _reason_formatters.buildReasonMessageForEqlAlert,
            ruleExecutionLogger,
            tuple,
            alertSuppression: completeRule.ruleParams.alertSuppression,
            wrapSuppressedHits,
            alertTimestampOverride,
            alertWithSuppression,
            experimentalFeatures
          });
        } else {
          newSignals = wrapHits(events, _reason_formatters.buildReasonMessageForEqlAlert);
        }
      } else if (sequences) {
        if (isAlertSuppressionActive) {
          result.warningMessages.push('Suppression is not supported for EQL sequence queries. The rule will proceed without suppression.');
        }
        newSignals = wrapSequences(sequences, _reason_formatters.buildReasonMessageForEqlAlert);
      } else {
        throw new Error('eql query response should have either `sequences` or `events` but had neither');
      }
      if ((_newSignals = newSignals) !== null && _newSignals !== void 0 && _newSignals.length) {
        const createResult = await bulkCreate(newSignals, undefined, (0, _enrichments.createEnrichEventsFunction)({
          services,
          logger: ruleExecutionLogger
        }));
        (0, _utils.addToSearchAfterReturn)({
          current: result,
          next: createResult
        });
      }
      if (response.hits.total && response.hits.total.value >= ruleParams.maxSignals) {
        const maxSignalsWarning = isAlertSuppressionActive && events !== null && events !== void 0 && events.length ? (0, _utils.getSuppressionMaxSignalsWarning)() : (0, _utils.getMaxSignalsWarning)();
        result.warningMessages.push(maxSignalsWarning);
      }
      return result;
    } catch (error) {
      if (typeof error.message === 'string' && error.message.includes('verification_exception')) {
        // We report errors that are more related to user configuration of rules rather than system outages as "user errors"
        // so SLO dashboards can show less noise around system outages
        result.userError = true;
      }
      result.errors.push(error.message);
      result.success = false;
      return result;
    }
  });
};
exports.eqlExecutor = eqlExecutor;