"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getActions = void 0;
var _esQuery = require("@kbn/es-query");
var _constants = require("../../../common/endpoint/constants");
var _utils = require("../services/actions/utils");
var _wrap_errors = require("./wrap_errors");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const queryOptions = Object.freeze({
  ignore: [404]
});
const getActionTypeFilter = actionType => {
  return actionType === 'manual' ? {
    must_not: {
      exists: {
        field: 'data.alert_id'
      }
    }
  } : actionType === 'automated' ? {
    filter: {
      exists: {
        field: 'data.alert_id'
      }
    }
  } : {};
};
const getActions = async ({
  agentTypes,
  commands,
  elasticAgentIds,
  esClient,
  endDate,
  from,
  size,
  startDate,
  userIds,
  unExpiredOnly,
  types
}) => {
  var _actionRequests$body, _actionRequests$body$;
  const additionalFilters = [];
  if (commands !== null && commands !== void 0 && commands.length) {
    additionalFilters.push({
      terms: {
        'data.command': commands
      }
    });
  }
  if (agentTypes !== null && agentTypes !== void 0 && agentTypes.length) {
    additionalFilters.push({
      terms: {
        input_type: agentTypes
      }
    });
  }
  if (elasticAgentIds !== null && elasticAgentIds !== void 0 && elasticAgentIds.length) {
    additionalFilters.push({
      terms: {
        agents: elasticAgentIds
      }
    });
  }
  if (unExpiredOnly) {
    additionalFilters.push({
      range: {
        expiration: {
          gte: 'now'
        }
      }
    });
  }
  const dateFilters = (0, _utils.getDateFilters)({
    startDate,
    endDate
  });
  const actionsFilters = [...dateFilters, ...additionalFilters];
  const must = [{
    bool: {
      filter: actionsFilters
    }
  }];
  if (userIds !== null && userIds !== void 0 && userIds.length) {
    const userIdsKql = userIds.map(userId => `user_id:${userId}`).join(' or ');
    const mustClause = (0, _esQuery.toElasticsearchQuery)((0, _esQuery.fromKueryExpression)(userIdsKql));
    must.push(mustClause);
  }
  const isNotASingleActionType = !types || types && types.length > 1;
  const actionsSearchQuery = {
    index: _constants.ENDPOINT_ACTIONS_INDEX,
    size,
    from,
    body: {
      query: {
        bool: {
          must,
          ...(isNotASingleActionType ? {} : getActionTypeFilter(types[0]))
        }
      },
      sort: [{
        '@timestamp': {
          order: 'desc'
        }
      }]
    }
  };
  const actionRequests = await esClient.search(actionsSearchQuery, {
    ...queryOptions,
    meta: true
  }).catch(_wrap_errors.catchAndWrapError);

  // only one type of actions
  const actionIds = actionRequests === null || actionRequests === void 0 ? void 0 : (_actionRequests$body = actionRequests.body) === null || _actionRequests$body === void 0 ? void 0 : (_actionRequests$body$ = _actionRequests$body.hits) === null || _actionRequests$body$ === void 0 ? void 0 : _actionRequests$body$.hits.map(e => {
    return e._source.EndpointActions.action_id;
  });
  return {
    actionIds,
    actionRequests
  };
};
exports.getActions = getActions;