"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getAgentStatus = exports.SENTINEL_ONE_NETWORK_STATUS = void 0;
var _constants = require("@kbn/stack-connectors-plugin/common/sentinelone/constants");
var _lodash = require("lodash");
var _stringify = require("../../utils/stringify");
var _types = require("../../../../common/endpoint/types");
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getAgentStatus = async ({
  agentType,
  agentIds,
  connectorActionsClient,
  logger
}) => {
  var _agentDetailsResponse;
  let connectorList = [];
  try {
    connectorList = await connectorActionsClient.getAll();
  } catch (err) {
    throw new _custom_http_request_error.CustomHttpRequestError(`Unable to retrieve list of stack connectors: ${err.message}`,
    // failure here is likely due to Authz, but because we don't have a good way to determine that,
    // the `statusCode` below is set to `400` instead of `401`.
    400, err);
  }
  const connector = connectorList.find(({
    actionTypeId,
    isDeprecated,
    isMissingSecrets
  }) => {
    return actionTypeId === _constants.SENTINELONE_CONNECTOR_ID && !isDeprecated && !isMissingSecrets;
  });
  if (!connector) {
    throw new _custom_http_request_error.CustomHttpRequestError(`No SentinelOne stack connector found`, 400, connectorList);
  }
  logger.debug(`Using SentinelOne stack connector: ${connector.name} (${connector.id})`);
  const agentDetailsResponse = await connectorActionsClient.execute({
    actionId: connector.id,
    params: {
      subAction: _constants.SUB_ACTION.GET_AGENTS,
      subActionParams: {
        uuids: agentIds.filter(agentId => agentId.trim().length).join(',')
      }
    }
  });
  if (agentDetailsResponse.status === 'error') {
    logger.error((0, _stringify.stringify)(agentDetailsResponse));
    throw new _custom_http_request_error.CustomHttpRequestError(`Attempt retrieve agent information from to SentinelOne failed: ${agentDetailsResponse.serviceMessage || agentDetailsResponse.message}`, 500, agentDetailsResponse);
  }
  const agentDetailsById = (0, _lodash.keyBy)((_agentDetailsResponse = agentDetailsResponse.data) === null || _agentDetailsResponse === void 0 ? void 0 : _agentDetailsResponse.data, 'uuid');
  logger.debug(`Response from SentinelOne API:\n${(0, _stringify.stringify)(agentDetailsById)}`);
  return agentIds.reduce((acc, agentId) => {
    const thisAgentDetails = agentDetailsById[agentId];
    const thisAgentStatus = {
      agentType,
      id: agentId,
      found: false,
      isolated: false,
      isPendingUninstall: false,
      isUninstalled: false,
      lastSeen: '',
      pendingActions: {
        execute: 0,
        upload: 0,
        unisolate: 0,
        isolate: 0,
        'get-file': 0,
        'kill-process': 0,
        'suspend-process': 0,
        'running-processes': 0
      },
      status: _types.HostStatus.UNENROLLED
    };
    if (thisAgentDetails) {
      (0, _lodash.merge)(thisAgentStatus, {
        found: true,
        lastSeen: thisAgentDetails.updatedAt,
        isPendingUninstall: thisAgentDetails.isPendingUninstall,
        isUninstalled: thisAgentDetails.isUninstalled,
        isolated: thisAgentDetails.networkStatus === SENTINEL_ONE_NETWORK_STATUS.DISCONNECTED,
        status: !thisAgentDetails.isActive ? _types.HostStatus.OFFLINE : _types.HostStatus.HEALTHY,
        pendingActions: {
          isolate: thisAgentDetails.networkStatus === SENTINEL_ONE_NETWORK_STATUS.DISCONNECTING ? 1 : 0,
          unisolate: thisAgentDetails.networkStatus === SENTINEL_ONE_NETWORK_STATUS.CONNECTING ? 1 : 0
        }
      });
    }
    acc[agentId] = thisAgentStatus;
    return acc;
  }, {});
};
exports.getAgentStatus = getAgentStatus;
let SENTINEL_ONE_NETWORK_STATUS = exports.SENTINEL_ONE_NETWORK_STATUS = /*#__PURE__*/function (SENTINEL_ONE_NETWORK_STATUS) {
  SENTINEL_ONE_NETWORK_STATUS["CONNECTING"] = "connecting";
  SENTINEL_ONE_NETWORK_STATUS["CONNECTED"] = "connected";
  SENTINEL_ONE_NETWORK_STATUS["DISCONNECTING"] = "disconnecting";
  SENTINEL_ONE_NETWORK_STATUS["DISCONNECTED"] = "disconnected";
  return SENTINEL_ONE_NETWORK_STATUS;
}({});