"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerResponseActionRoutes = registerResponseActionRoutes;
var _constants = require("../../services/actions/constants");
var _stringify = require("../../utils/stringify");
var _services = require("../../services");
var _custom_http_request_error = require("../../../utils/custom_http_request_error");
var _endpoint = require("../../../../common/api/endpoint");
var _constants2 = require("../../../../common/endpoint/constants");
var _with_endpoint_authz = require("../with_endpoint_authz");
var _error_handler = require("../error_handler");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

function registerResponseActionRoutes(router, endpointContext) {
  const logger = endpointContext.logFactory.get('hostIsolation');

  /**
   * @deprecated use ISOLATE_HOST_ROUTE_V2 instead
   */
  router.versioned.post({
    access: 'public',
    path: _constants2.ISOLATE_HOST_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.IsolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canIsolateHost']
  }, logger, redirectHandler(_constants2.ISOLATE_HOST_ROUTE_V2)));

  /**
   * @deprecated use RELEASE_HOST_ROUTE instead
   */
  router.versioned.post({
    access: 'public',
    path: _constants2.UNISOLATE_HOST_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UnisolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canUnIsolateHost']
  }, logger, redirectHandler(_constants2.UNISOLATE_HOST_ROUTE_V2)));
  router.versioned.post({
    access: 'public',
    path: _constants2.ISOLATE_HOST_ROUTE_V2,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.IsolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canIsolateHost']
  }, logger, responseActionRequestHandler(endpointContext, 'isolate')));
  router.versioned.post({
    access: 'public',
    path: _constants2.UNISOLATE_HOST_ROUTE_V2,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UnisolateRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canUnIsolateHost']
  }, logger, responseActionRequestHandler(endpointContext, 'unisolate')));
  router.versioned.post({
    access: 'public',
    path: _constants2.KILL_PROCESS_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.KillProcessRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canKillProcess']
  }, logger, responseActionRequestHandler(endpointContext, 'kill-process')));
  router.versioned.post({
    access: 'public',
    path: _constants2.SUSPEND_PROCESS_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.SuspendProcessRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canSuspendProcess']
  }, logger, responseActionRequestHandler(endpointContext, 'suspend-process')));
  router.versioned.post({
    access: 'public',
    path: _constants2.GET_PROCESSES_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.GetProcessesRouteRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canGetRunningProcesses']
  }, logger, responseActionRequestHandler(endpointContext, 'running-processes')));
  router.versioned.post({
    access: 'public',
    path: _constants2.GET_FILE_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.EndpointActionGetFileSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteFileOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'get-file')));
  router.versioned.post({
    access: 'public',
    path: _constants2.EXECUTE_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution']
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.ExecuteActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteExecuteOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'execute')));
  router.versioned.post({
    access: 'public',
    path: _constants2.UPLOAD_ROUTE,
    options: {
      authRequired: true,
      tags: ['access:securitySolution'],
      body: {
        accepts: ['multipart/form-data'],
        output: 'stream',
        maxBytes: endpointContext.serverConfig.maxUploadResponseActionFileBytes
      }
    }
  }).addVersion({
    version: '2023-10-31',
    validate: {
      request: _endpoint.UploadActionRequestSchema
    }
  }, (0, _with_endpoint_authz.withEndpointAuthz)({
    all: ['canWriteFileOperations']
  }, logger, responseActionRequestHandler(endpointContext, 'upload')));
}
function responseActionRequestHandler(endpointContext, command) {
  const logger = endpointContext.logFactory.get('responseActionsHandler');
  return async (context, req, res) => {
    var _endpointContext$serv;
    logger.debug(`response action [${command}]:\n${(0, _stringify.stringify)(req.body)}`);

    // Note:  because our API schemas are defined as module static variables (as opposed to a
    //        `getter` function), we need to include this additional validation here, since
    //        `agent_type` is included in the schema independent of the feature flag
    if (req.body.agent_type === 'sentinel_one' && !endpointContext.experimentalFeatures.responseActionsSentinelOneV1Enabled) {
      return (0, _error_handler.errorHandler)(logger, res, new _custom_http_request_error.CustomHttpRequestError(`[request body.agent_type]: feature is disabled`, 400));
    }
    const user = (_endpointContext$serv = endpointContext.service.security) === null || _endpointContext$serv === void 0 ? void 0 : _endpointContext$serv.authc.getCurrentUser(req);
    const esClient = (await context.core).elasticsearch.client.asInternalUser;
    const casesClient = await endpointContext.service.getCasesClient(req);
    const connectorActions = (await context.actions).getActionsClient();
    const responseActionsClient = (0, _services.getResponseActionsClient)(req.body.agent_type || 'endpoint', {
      esClient,
      casesClient,
      endpointService: endpointContext.service,
      username: (user === null || user === void 0 ? void 0 : user.username) || 'unknown',
      connectorActions: new _services.NormalizedExternalConnectorClient(connectorActions, logger)
    });
    try {
      var _ref;
      let action;
      switch (command) {
        case 'isolate':
          action = await responseActionsClient.isolate(req.body);
          break;
        case 'unisolate':
          action = await responseActionsClient.release(req.body);
          break;
        case 'running-processes':
          action = await responseActionsClient.runningProcesses(req.body);
          break;
        case 'execute':
          action = await responseActionsClient.execute(req.body);
          break;
        case 'suspend-process':
          action = await responseActionsClient.suspendProcess(req.body);
          break;
        case 'kill-process':
          action = await responseActionsClient.killProcess(req.body);
          break;
        case 'get-file':
          action = await responseActionsClient.getFile(req.body);
          break;
        case 'upload':
          action = await responseActionsClient.upload(req.body);
          break;
        default:
          throw new _custom_http_request_error.CustomHttpRequestError(`No handler found for response action command: [${command}]`, 501);
      }
      const {
        action: actionId,
        ...data
      } = action;

      // `action` is deprecated, but still returned in order to ensure backwards compatibility
      const legacyResponseData = _constants.responseActionsWithLegacyActionProperty.includes(command) ? {
        action: (_ref = actionId !== null && actionId !== void 0 ? actionId : data.id) !== null && _ref !== void 0 ? _ref : ''
      } : {};
      return res.ok({
        body: {
          ...legacyResponseData,
          data
        }
      });
    } catch (err) {
      return (0, _error_handler.errorHandler)(logger, res, err);
    }
  };
}
function redirectHandler(location) {
  return async (context, _req, res) => {
    const basePath = (await context.securitySolution).getServerBasePath();
    return res.custom({
      statusCode: 308,
      headers: {
        location: `${basePath}${location}`
      }
    });
  };
}