"use strict";

var _interopRequireDefault = require("@babel/runtime/helpers/interopRequireDefault");
Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.CompleteExternalActionsTaskRunner = void 0;
var _defineProperty2 = _interopRequireDefault(require("@babel/runtime/helpers/defineProperty"));
var _errors = require("../../services/actions/clients/errors");
var _utils = require("../../utils");
var _stringify = require("../../utils/stringify");
var _constants = require("../../../../common/endpoint/service/response_actions/constants");
var _queue_processor = require("../../utils/queue_processor");
var _constants2 = require("../../../../common/endpoint/constants");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

/**
 * A task manager runner responsible for checking the status of and completing pending actions
 * that were sent to 3rd party EDR systems.
 */
class CompleteExternalActionsTaskRunner {
  constructor(endpointContextServices, esClient, nextRunInterval = '60s', taskId, taskType) {
    (0, _defineProperty2.default)(this, "log", void 0);
    (0, _defineProperty2.default)(this, "updatesQueue", void 0);
    (0, _defineProperty2.default)(this, "abortController", new AbortController());
    (0, _defineProperty2.default)(this, "errors", []);
    this.endpointContextServices = endpointContextServices;
    this.esClient = esClient;
    this.nextRunInterval = nextRunInterval;
    this.taskId = taskId;
    this.taskType = taskType;
    this.log = this.endpointContextServices.createLogger(
    // Adding a unique identifier to the end of the class name to help identify log entries related to this run
    `${this.constructor.name}.${Math.random().toString(32).substring(2, 8)}`);
    this.updatesQueue = new _queue_processor.QueueProcessor({
      batchHandler: this.queueBatchProcessor.bind(this),
      batchSize: 50,
      logger: this.log
    });
  }
  async queueBatchProcessor({
    batch,
    data
  }) {
    const operations = [];
    for (const actionResponseDoc of data) {
      operations.push({
        create: {
          _index: _constants2.ENDPOINT_ACTION_RESPONSES_INDEX
        }
      }, actionResponseDoc);
    }
    const bulkResponse = await this.esClient.bulk({
      index: _constants2.ENDPOINT_ACTION_RESPONSES_INDEX,
      operations
    }).catch(_utils.catchAndWrapError);
    if (bulkResponse.errors) {
      this.errors.push(`Batch [${batch}] processing of [${data.length}] items generated the following errors:\n${(0, _stringify.stringify)(bulkResponse)}`);
    }
  }
  getNextRunDate() {
    const nextRun = new Date();
    const nextRunInterval = this.nextRunInterval;
    if (nextRunInterval.endsWith('s')) {
      const seconds = parseInt(nextRunInterval.slice(0, -1), 10);
      nextRun.setSeconds(nextRun.getSeconds() + seconds);
    } else if (nextRunInterval.endsWith('m')) {
      const minutes = parseInt(nextRunInterval.slice(0, -1), 10);
      nextRun.setMinutes(nextRun.getMinutes() + minutes);
    } else {
      this.log.error(`Invalid task interval: ${nextRunInterval}`);
      return;
    }
    return nextRun;
  }
  async run() {
    this.log.debug(`Started: Checking status of external response actions`);
    this.abortController = new AbortController();

    // If license is not `enterprise`, then exit. Support for external response actions is a
    // Enterprise level feature.
    if (!this.endpointContextServices.getLicenseService().isEnterprise()) {
      this.abortController.abort(`License not Enterprise!`);
      this.log.debug(`Exiting: Run aborted due to license not being Enterprise`);
      return;
    }

    // Collect update needed to complete response actions
    await Promise.all(_constants.RESPONSE_ACTION_AGENT_TYPE.filter(agentType => agentType !== 'endpoint').map(agentType => {
      // If run was aborted, then stop looping through
      if (this.abortController.signal.aborted) {
        return null;
      }
      const agentTypeActionsClient = this.endpointContextServices.getInternalResponseActionsClient({
        agentType,
        taskType: this.taskType,
        taskId: this.taskId
      });
      return agentTypeActionsClient.processPendingActions({
        abortSignal: this.abortController.signal,
        addToQueue: this.updatesQueue.addToQueue.bind(this.updatesQueue)
      }).catch(err => {
        // ignore errors due to connector not being configured - no point in logging errors if a customer
        // is not using response actions for the given agent type
        if (err instanceof _errors.ResponseActionsConnectorNotConfiguredError) {
          this.log.debug(`Skipping agentType [${agentType}]: No stack connector configured for this agent type`);
          return null;
        }
        this.errors.push(err.message);
      });
    }));
    await this.updatesQueue.complete();
    this.abortController.abort(`Run complete.`);
    if (this.errors.length) {
      this.log.error(`${this.errors.length} errors were encountered while running task:\n${this.errors.join('\n')}`);
    }
    this.log.debug(`Completed: Checking status of external response actions`);
    return {
      state: {},
      runAt: this.getNextRunDate()
    };
  }
  async cancel() {
    if (!this.abortController.signal.aborted) {
      this.abortController.abort('Task runner canceled!');

      // Sleep 2 seconds to give an opportunity for the abort signal to be processed
      await new Promise(r => setTimeout(r, 2000));

      // Wait for remainder of updates to be written to ES
      await this.updatesQueue.complete();
    }
  }
}
exports.CompleteExternalActionsTaskRunner = CompleteExternalActionsTaskRunner;