"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.ATTACK_DISCOVERY_TOOL_DESCRIPTION = exports.ATTACK_DISCOVERY_TOOL = void 0;
var _prompts = require("@langchain/core/prompts");
var _helpers = require("@kbn/elastic-assistant-plugin/server/lib/langchain/helpers");
var _chains = require("langchain/chains");
var _output_parsers = require("langchain/output_parsers");
var _tools = require("@langchain/core/tools");
var _common = require("../../../../common");
var _get_anonymized_alerts = require("./get_anonymized_alerts");
var _get_output_parser = require("./get_output_parser");
var _helpers2 = require("../open_and_acknowledged_alerts/helpers");
var _get_attack_discovery_prompt = require("./get_attack_discovery_prompt");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const ATTACK_DISCOVERY_TOOL_DESCRIPTION = exports.ATTACK_DISCOVERY_TOOL_DESCRIPTION = 'Call this for attack discoveries containing `markdown` that should be displayed verbatim (with no additional processing).';

/**
 * Returns a tool for generating attack discoveries from open and acknowledged
 * alerts, or null if the request doesn't have all the required parameters.
 */
const ATTACK_DISCOVERY_TOOL = exports.ATTACK_DISCOVERY_TOOL = {
  id: 'attack-discovery',
  name: 'AttackDiscoveryTool',
  description: ATTACK_DISCOVERY_TOOL_DESCRIPTION,
  sourceRegister: _common.APP_UI_ID,
  isSupported: params => {
    const {
      alertsIndexPattern,
      llm,
      request,
      size
    } = params;
    return (0, _helpers.requestHasRequiredAnonymizationParams)(request) && alertsIndexPattern != null && size != null && !(0, _helpers2.sizeIsOutOfRange)(size) && llm != null;
  },
  getTool(params) {
    if (!this.isSupported(params)) return null;
    const {
      alertsIndexPattern,
      anonymizationFields,
      esClient,
      langChainTimeout,
      llm,
      onNewReplacements,
      replacements,
      size
    } = params;
    return new _tools.DynamicTool({
      name: 'AttackDiscoveryTool',
      description: ATTACK_DISCOVERY_TOOL_DESCRIPTION,
      func: async () => {
        if (llm == null) {
          throw new Error('LLM is required for attack discoveries');
        }
        const anonymizedAlerts = await (0, _get_anonymized_alerts.getAnonymizedAlerts)({
          alertsIndexPattern,
          anonymizationFields,
          esClient,
          onNewReplacements,
          replacements,
          size
        });
        const alertsContextCount = anonymizedAlerts.length;
        if (alertsContextCount === 0) {
          // No alerts to analyze, so return an empty attack discoveries array
          return JSON.stringify({
            alertsContextCount,
            attackDiscoveries: []
          }, null, 2);
        }
        const outputParser = (0, _get_output_parser.getOutputParser)();
        const outputFixingParser = _output_parsers.OutputFixingParser.fromLLM(llm, outputParser);
        const prompt = new _prompts.PromptTemplate({
          template: `Answer the user's question as best you can:\n{format_instructions}\n{query}`,
          inputVariables: ['query'],
          partialVariables: {
            format_instructions: outputFixingParser.getFormatInstructions()
          }
        });
        const answerFormattingChain = new _chains.LLMChain({
          llm,
          prompt,
          outputKey: 'records',
          outputParser: outputFixingParser
        });
        const result = await answerFormattingChain.call({
          query: (0, _get_attack_discovery_prompt.getAttackDiscoveryPrompt)({
            anonymizedAlerts
          }),
          timeout: langChainTimeout
        });
        const attackDiscoveries = result.records;
        return JSON.stringify({
          alertsContextCount,
          attackDiscoveries
        }, null, 2);
      },
      tags: ['attack-discovery']
    });
  }
};