"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.getEndpointConsoleCommands = exports.HELP_GROUPS = void 0;
var _i18n = require("@kbn/i18n");
var _utils = require("../../../../../common/endpoint/service/response_actions/utils");
var _upload_action = require("../command_render_components/upload_action");
var _console_argument_selectors = require("../../console_argument_selectors");
var _experimental_features_service = require("../../../../common/experimental_features_service");
var _constants = require("../../../../../common/endpoint/service/response_actions/constants");
var _get_file_action = require("../command_render_components/get_file_action");
var _isolate_action = require("../command_render_components/isolate_action");
var _release_action = require("../command_render_components/release_action");
var _kill_process_action = require("../command_render_components/kill_process_action");
var _suspend_process_action = require("../command_render_components/suspend_process_action");
var _status_action = require("../command_render_components/status_action");
var _get_processes_action = require("../command_render_components/get_processes_action");
var _execute_action = require("../command_render_components/execute_action");
var _translations = require("../../../../common/translations");
var _get_command_about_info = require("./get_command_about_info");
var _utils2 = require("./utils");
var _translations2 = require("../../../common/translations");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const emptyArgumentValidator = argData => {
  var _argData$;
  if ((argData === null || argData === void 0 ? void 0 : argData.length) > 0 && typeof argData[0] === 'string' && ((_argData$ = argData[0]) === null || _argData$ === void 0 ? void 0 : _argData$.trim().length) > 0) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.emptyArgumentMessage', {
      defaultMessage: 'Argument cannot be empty'
    });
  }
};
const pidValidator = argData => {
  const emptyResult = emptyArgumentValidator(argData);
  if (emptyResult !== true) {
    return emptyResult;
  } else if (Number.isSafeInteger(Number(argData)) && Number(argData) > 0) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.invalidPidMessage', {
      defaultMessage: 'Argument must be a positive number representing the PID of a process'
    });
  }
};
const executeTimeoutValidator = argData => {
  if (String(argData).trim().length && (0, _utils2.validateUnitOfTime)(String(argData).trim())) {
    return true;
  } else {
    return _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.invalidExecuteTimeout', {
      defaultMessage: 'Argument must be a string with a positive integer value followed by a unit of time (h for hours, m for minutes, s for seconds). Example: 37m.'
    });
  }
};
const capabilitiesAndPrivilegesValidator = agentType => {
  return command => {
    const privileges = command.commandDefinition.meta.privileges;
    const agentCapabilities = command.commandDefinition.meta.capabilities;
    const commandName = command.commandDefinition.name;
    const responderCapability = _constants.RESPONSE_CONSOLE_ACTION_COMMANDS_TO_ENDPOINT_CAPABILITY[commandName];
    let errorMessage = '';
    if (!responderCapability) {
      errorMessage = errorMessage.concat((0, _translations.UPGRADE_AGENT_FOR_RESPONDER)(agentType, commandName));
    }
    if (responderCapability) {
      if (!agentCapabilities.includes(responderCapability)) {
        errorMessage = errorMessage.concat((0, _translations.UPGRADE_AGENT_FOR_RESPONDER)(agentType, commandName));
      }
    }
    if (!(0, _utils.getRbacControl)({
      commandName,
      privileges
    })) {
      errorMessage = errorMessage.concat(_translations.INSUFFICIENT_PRIVILEGES_FOR_COMMAND);
    }
    if (errorMessage.length) {
      return errorMessage;
    }
    return true;
  };
};
const HELP_GROUPS = exports.HELP_GROUPS = Object.freeze({
  responseActions: {
    position: 0,
    label: _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.groups.responseActions', {
      defaultMessage: 'Response actions'
    })
  }
});
const ENTER_PID_OR_ENTITY_ID_INSTRUCTION = _i18n.i18n.translate('xpack.securitySolution.endpointResponseActionsConsoleCommands.enterPidOrEntityId', {
  defaultMessage: 'Enter a pid or an entity id to execute'
});
const ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION = _i18n.i18n.translate('xpack.securitySolution.endpointResponseActionsConsoleCommands.enterOrAddOptionalComment', {
  defaultMessage: 'Hit enter to execute or add an optional comment'
});
const COMMENT_ARG_ABOUT = _i18n.i18n.translate('xpack.securitySolution.endpointConsoleCommands.suspendProcess.commandArgAbout', {
  defaultMessage: 'A comment to go along with the action'
});
const getEndpointConsoleCommands = ({
  endpointAgentId,
  agentType,
  endpointCapabilities,
  endpointPrivileges
}) => {
  const featureFlags = _experimental_features_service.ExperimentalFeaturesService.get();
  const isUploadEnabled = featureFlags.responseActionUploadEnabled;
  const doesEndpointSupportCommand = commandName => {
    const responderCapability = _constants.RESPONSE_CONSOLE_ACTION_COMMANDS_TO_ENDPOINT_CAPABILITY[commandName];
    if (responderCapability) {
      return endpointCapabilities.includes(responderCapability);
    }
    return false;
  };
  const consoleCommands = [{
    name: 'isolate',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.isolate.about,
      isSupported: doesEndpointSupportCommand('isolate')
    }),
    RenderComponent: _isolate_action.IsolateActionResult,
    meta: {
      agentType,
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'isolate --comment "isolate this host"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 0,
    helpDisabled: doesEndpointSupportCommand('isolate') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'isolate',
      privileges: endpointPrivileges
    })
  }, {
    name: 'release',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.release.about,
      isSupported: doesEndpointSupportCommand('release')
    }),
    RenderComponent: _release_action.ReleaseActionResult,
    meta: {
      agentType,
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'release --comment "release this host"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 1,
    helpDisabled: doesEndpointSupportCommand('release') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'release',
      privileges: endpointPrivileges
    })
  }, {
    //
    name: 'kill-process',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.killProcess.about,
      isSupported: doesEndpointSupportCommand('kill-process')
    }),
    RenderComponent: _kill_process_action.KillProcessActionResult,
    meta: {
      agentType,
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'kill-process --pid 123 --comment "kill this process"',
    exampleInstruction: ENTER_PID_OR_ENTITY_ID_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      },
      pid: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _translations2.CONSOLE_COMMANDS.killProcess.args.pid.about,
        validate: pidValidator
      },
      entityId: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _translations2.CONSOLE_COMMANDS.killProcess.args.entityId.about,
        validate: emptyArgumentValidator
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 4,
    helpDisabled: doesEndpointSupportCommand('kill-process') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'kill-process',
      privileges: endpointPrivileges
    })
  }, {
    name: 'suspend-process',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.suspendProcess.about,
      isSupported: doesEndpointSupportCommand('suspend-process')
    }),
    RenderComponent: _suspend_process_action.SuspendProcessActionResult,
    meta: {
      agentType,
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'suspend-process --pid 123 --comment "suspend this process"',
    exampleInstruction: ENTER_PID_OR_ENTITY_ID_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      },
      pid: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _translations2.CONSOLE_COMMANDS.suspendProcess.args.pid.about,
        validate: pidValidator
      },
      entityId: {
        required: false,
        allowMultiples: false,
        exclusiveOr: true,
        about: _translations2.CONSOLE_COMMANDS.suspendProcess.args.entityId.about,
        validate: emptyArgumentValidator
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 5,
    helpDisabled: doesEndpointSupportCommand('suspend-process') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'suspend-process',
      privileges: endpointPrivileges
    })
  }, {
    name: 'status',
    about: _translations2.CONSOLE_COMMANDS.status.about,
    RenderComponent: _status_action.EndpointStatusActionResult,
    meta: {
      endpointId: endpointAgentId
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 2
  }, {
    name: 'processes',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.processes.about,
      isSupported: doesEndpointSupportCommand('processes')
    }),
    RenderComponent: _get_processes_action.GetProcessesActionResult,
    meta: {
      agentType,
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'processes --comment "get the processes"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    args: {
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 3,
    helpDisabled: doesEndpointSupportCommand('processes') === false,
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'processes',
      privileges: endpointPrivileges
    })
  }, {
    name: 'get-file',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.getFile.about,
      isSupported: doesEndpointSupportCommand('processes')
    }),
    RenderComponent: _get_file_action.GetFileActionResult,
    meta: {
      agentType,
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'get-file --path "/full/path/to/file.txt" --comment "Possible malware"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      path: {
        required: true,
        allowMultiples: false,
        about: _translations2.CONSOLE_COMMANDS.getFile.args.path.about,
        validate: argData => {
          return emptyArgumentValidator(argData);
        }
      },
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 6,
    helpDisabled: !doesEndpointSupportCommand('get-file'),
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'get-file',
      privileges: endpointPrivileges
    })
  }, {
    name: 'execute',
    about: (0, _get_command_about_info.getCommandAboutInfo)({
      aboutInfo: _translations2.CONSOLE_COMMANDS.execute.about,
      isSupported: doesEndpointSupportCommand('execute')
    }),
    RenderComponent: _execute_action.ExecuteActionResult,
    meta: {
      agentType,
      endpointId: endpointAgentId,
      capabilities: endpointCapabilities,
      privileges: endpointPrivileges
    },
    exampleUsage: 'execute --command "ls -al" --timeout 2s --comment "Get list of all files"',
    exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
    validate: capabilitiesAndPrivilegesValidator(agentType),
    mustHaveArgs: true,
    args: {
      command: {
        required: true,
        allowMultiples: false,
        about: (0, _execute_action.getExecuteCommandArgAboutInfo)(),
        mustHaveValue: 'non-empty-string'
      },
      timeout: {
        required: false,
        allowMultiples: false,
        about: _translations2.CONSOLE_COMMANDS.execute.args.timeout.about,
        mustHaveValue: 'non-empty-string',
        validate: executeTimeoutValidator
      },
      comment: {
        required: false,
        allowMultiples: false,
        about: COMMENT_ARG_ABOUT
      }
    },
    helpGroupLabel: HELP_GROUPS.responseActions.label,
    helpGroupPosition: HELP_GROUPS.responseActions.position,
    helpCommandPosition: 6,
    helpDisabled: !doesEndpointSupportCommand('execute'),
    helpHidden: !(0, _utils.getRbacControl)({
      commandName: 'execute',
      privileges: endpointPrivileges
    })
  }];

  // `upload` command
  // planned for 8.9
  if (isUploadEnabled) {
    consoleCommands.push({
      name: 'upload',
      about: (0, _get_command_about_info.getCommandAboutInfo)({
        aboutInfo: _translations2.CONSOLE_COMMANDS.upload.about,
        isSupported: doesEndpointSupportCommand('upload')
      }),
      RenderComponent: _upload_action.UploadActionResult,
      meta: {
        agentType,
        endpointId: endpointAgentId,
        capabilities: endpointCapabilities,
        privileges: endpointPrivileges
      },
      exampleUsage: 'upload --file --overwrite --comment "script to fix registry"',
      exampleInstruction: ENTER_OR_ADD_COMMENT_ARG_INSTRUCTION,
      validate: capabilitiesAndPrivilegesValidator(agentType),
      mustHaveArgs: true,
      args: {
        file: {
          required: true,
          allowMultiples: false,
          about: _translations2.CONSOLE_COMMANDS.upload.args.file.about,
          mustHaveValue: 'truthy',
          SelectorComponent: _console_argument_selectors.ArgumentFileSelector
        },
        overwrite: {
          required: false,
          allowMultiples: false,
          about: _translations2.CONSOLE_COMMANDS.upload.args.overwrite.about,
          mustHaveValue: false
        },
        comment: {
          required: false,
          allowMultiples: false,
          mustHaveValue: 'non-empty-string',
          about: COMMENT_ARG_ABOUT
        }
      },
      helpGroupLabel: HELP_GROUPS.responseActions.label,
      helpGroupPosition: HELP_GROUPS.responseActions.position,
      helpCommandPosition: 7,
      helpDisabled: !doesEndpointSupportCommand('upload'),
      helpHidden: !(0, _utils.getRbacControl)({
        commandName: 'upload',
        privileges: endpointPrivileges
      })
    });
  }
  return consoleCommands;
};
exports.getEndpointConsoleCommands = getEndpointConsoleCommands;