"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.useResponderActionData = void 0;
var _react = require("react");
var _use_experimental_features = require("../../../common/hooks/use_experimental_features");
var _sentinelone_alert_check = require("../../../common/utils/sentinelone_alert_check");
var _hooks = require("../../../management/hooks");
var _types = require("../../../../common/endpoint/types");
var _translations = require("./translations");
var _helpers = require("../host_isolation/helpers");
/*
 * Copyright Elasticsearch B.V. and/or licensed to Elasticsearch B.V. under one
 * or more contributor license agreements. Licensed under the Elastic License
 * 2.0; you may not use this file except in compliance with the Elastic License
 * 2.0.
 */

const getThirdPartyAgentInfo = eventData => {
  return {
    agent: {
      id: (0, _sentinelone_alert_check.getSentinelOneAgentId)(eventData) || '',
      type: (0, _helpers.getFieldValue)({
        category: 'event',
        field: 'event.module'
      }, eventData)
    },
    host: {
      name: (0, _helpers.getFieldValue)({
        category: 'host',
        field: 'host.name'
      }, eventData),
      os: {
        name: (0, _helpers.getFieldValue)({
          category: 'host',
          field: 'host.os.name'
        }, eventData),
        family: (0, _helpers.getFieldValue)({
          category: 'host',
          field: 'host.os.family'
        }, eventData),
        version: (0, _helpers.getFieldValue)({
          category: 'host',
          field: 'host.os.version'
        }, eventData)
      }
    },
    lastCheckin: (0, _helpers.getFieldValue)({
      category: 'kibana',
      field: 'kibana.alert.last_detected'
    }, eventData)
  };
};

/**
 * This hook is used to get the data needed to show the context menu items for the responder
 * actions.
 * @param endpointId the id of the endpoint
 * @param onClick the callback to handle the click event
 * @param agentType the type of agent, defaults to 'endpoint'
 * @param eventData the event data, exists only when agentType !== 'endpoint'
 * @returns an object with the data needed to show the context menu item
 */

const useResponderActionData = ({
  endpointId,
  onClick,
  agentType = 'endpoint',
  eventData
}) => {
  const isEndpointHost = agentType === 'endpoint';
  const showResponseActionsConsole = (0, _hooks.useWithShowResponder)();
  const isSentinelOneV1Enabled = (0, _use_experimental_features.useIsExperimentalFeatureEnabled)('responseActionsSentinelOneV1Enabled');
  const {
    data: hostInfo,
    isFetching,
    error
  } = (0, _hooks.useGetEndpointDetails)(endpointId, {
    enabled: Boolean(endpointId && isEndpointHost)
  });
  const [isDisabled, tooltip] = (0, _react.useMemo)(() => {
    var _error$body, _error$body2, _error$body3;
    // v8.13 disabled for third-party agent alerts if the feature flag is not enabled
    if (!isEndpointHost) {
      switch (agentType) {
        case 'sentinel_one':
          // Disable it if feature flag is disabled
          if (!isSentinelOneV1Enabled) {
            return [true, undefined];
          }
          // Event must have the property that identifies the agent id
          if (!(0, _sentinelone_alert_check.getSentinelOneAgentId)(eventData !== null && eventData !== void 0 ? eventData : null)) {
            return [true, _translations.SENTINEL_ONE_AGENT_ID_PROPERTY_MISSING];
          }
          return [false, undefined];
        default:
          return [true, undefined];
      }
    }

    // Still loading host info
    if (isFetching) {
      return [true, _translations.LOADING_ENDPOINT_DATA_TOOLTIP];
    }

    // if we got an error, and it's a 404, it means the endpoint is not from the endpoint host
    if (error && ((_error$body = error.body) === null || _error$body === void 0 ? void 0 : _error$body.statusCode) === 404) {
      return [true, _translations.NOT_FROM_ENDPOINT_HOST_TOOLTIP];
    }

    // if we got an error and,
    // it's a 400 with unenrolled in the error message (alerts can exist for endpoint that are no longer around)
    // or,
    // the Host status is `unenrolled`
    if (error && ((_error$body2 = error.body) === null || _error$body2 === void 0 ? void 0 : _error$body2.statusCode) === 400 && (_error$body3 = error.body) !== null && _error$body3 !== void 0 && _error$body3.message.includes('unenrolled') || (hostInfo === null || hostInfo === void 0 ? void 0 : hostInfo.host_status) === _types.HostStatus.UNENROLLED) {
      return [true, _translations.HOST_ENDPOINT_UNENROLLED_TOOLTIP];
    }

    // return general error tooltip
    if (error) {
      return [true, _translations.METADATA_API_ERROR_TOOLTIP];
    }
    return [false, undefined];
  }, [isEndpointHost, isFetching, error, hostInfo === null || hostInfo === void 0 ? void 0 : hostInfo.host_status, agentType, isSentinelOneV1Enabled, eventData]);
  const handleResponseActionsClick = (0, _react.useCallback)(() => {
    if (!isEndpointHost) {
      const agentInfoFromAlert = getThirdPartyAgentInfo(eventData || null);
      showResponseActionsConsole({
        agentId: agentInfoFromAlert.agent.id,
        agentType,
        capabilities: ['isolation'],
        hostName: agentInfoFromAlert.host.name,
        platform: agentInfoFromAlert.host.os.family
      });
    }
    if (hostInfo) {
      var _hostInfo$metadata$En;
      showResponseActionsConsole({
        agentId: hostInfo.metadata.agent.id,
        agentType: 'endpoint',
        capabilities: (_hostInfo$metadata$En = hostInfo.metadata.Endpoint.capabilities) !== null && _hostInfo$metadata$En !== void 0 ? _hostInfo$metadata$En : [],
        hostName: hostInfo.metadata.host.name
      });
    }
    if (onClick) onClick();
  }, [isEndpointHost, hostInfo, onClick, eventData, showResponseActionsConsole, agentType]);
  return {
    handleResponseActionsClick,
    isDisabled,
    tooltip
  };
};
exports.useResponderActionData = useResponderActionData;